/**
 * Right Panel Component
 * פאנל ימני - רשימות חכמות ורשימות משתמש
 */

'use client';

import { useState, useEffect } from 'react';
import { useListsStore } from '@/src/store/listsStore';
import { useNotesStore } from '@/src/store/notesStore';
import { smartLists } from '@/src/lib/notes-helpers';
import ListItem from './ListItem';
import CreateListDialog from './CreateListDialog';
import { Plus, Download, Upload } from 'lucide-react';
import { ScrollArea } from '@/components/ui/scroll-area';
import {
    DndContext,
    closestCenter,
    KeyboardSensor,
    PointerSensor,
    useSensor,
    useSensors,
    DragEndEvent,
} from '@dnd-kit/core';
import {
    arrayMove,
    SortableContext,
    sortableKeyboardCoordinates,
    useSortable,
    verticalListSortingStrategy,
} from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';
import toast from 'react-hot-toast';

// Sortable List Item Wrapper
function SortableListItem(props: any) {
    const {
        attributes,
        listeners,
        setNodeRef,
        transform,
        transition,
        isDragging,
    } = useSortable({ id: props.id });

    const style = {
        transform: CSS.Transform.toString(transform),
        transition,
        opacity: isDragging ? 0.5 : 1,
    };

    return (
        <div ref={setNodeRef} style={style}>
            <ListItem
                {...props}
                isDraggable
                dragHandleProps={{ ...attributes, ...listeners }}
            />
        </div>
    );
}

export default function RightPanel() {
    const { lists, selectedListId, selectList, deleteListApi, reorderLists, updateListApi, fetchLists } = useListsStore();
    const { notes } = useNotesStore();
    const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);

    // Ensure lists are loaded
    useEffect(() => {
        if (lists.length === 0) {
            fetchLists();
        }
    }, [lists.length, fetchLists]);

    const sensors = useSensors(
        useSensor(PointerSensor),
        useSensor(KeyboardSensor, {
            coordinateGetter: sortableKeyboardCoordinates,
        })
    );

    // Count notes for each smart list
    const smartListCounts = {
        important: notes.filter(n => n.important && !n.done).length,
        planned: notes.filter(n => n.dueDate && !n.done).length,
        all: notes.length,
        completed: notes.filter(n => n.done).length,
        flagged: notes.filter(n => n.flagged).length,
        today: notes.filter(n => {
            const today = new Date().toISOString().split('T')[0];
            return (n.reminderDate === today || n.dueDate === today) && !n.done;
        }).length,
    };

    // Count notes for each user list
    const getListCount = (listId: string) => {
        return notes.filter(n => n.listId === listId && !n.done).length;
    };

    // Handle drag end
    const handleDragEnd = async (event: DragEndEvent) => {
        const { active, over } = event;

        if (!over || active.id === over.id) return;

        const oldIndex = lists.findIndex((list) => list.id === active.id);
        const newIndex = lists.findIndex((list) => list.id === over.id);

        if (oldIndex === -1 || newIndex === -1) return;

        // Update local state immediately
        reorderLists(oldIndex, newIndex);

        // Update each list's sortOrder in the API
        const reorderedLists = arrayMove(lists, oldIndex, newIndex);

        try {
            await Promise.all(
                reorderedLists.map((list, index) =>
                    updateListApi(list.id, { sortOrder: index })
                )
            );
            toast.success('הסדר עודכן! ✅');
        } catch (error) {
            console.error('Error updating list order:', error);
            toast.error('שגיאה בעדכון הסדר');
        }
    };

    // Handle delete list
    const handleDeleteList = async (listId: string) => {
        try {
            await deleteListApi(listId);
            toast.success('הרשימה נמחקה בהצלחה! ✅');
        } catch (error) {
            console.error('Error deleting list:', error);
            toast.error('שגיאה במחיקת הרשימה');
        }
    };

    return (
        <div className="h-full flex flex-col bg-gradient-to-b from-purple-50 to-purple-100 rounded-r-2xl lg:rounded-2xl p-5 shadow-lg">
            <ScrollArea className="flex-1">
                {/* Smart Lists */}
                <div className="mb-6">
                    <h3 className="text-sm font-semibold text-purple-900 mb-3 px-2">
                        רשימות חכמות
                    </h3>
                    <div className="space-y-1">
                        {smartLists.map((smartList) => (
                            <ListItem
                                key={smartList.id}
                                id={smartList.id}
                                name={smartList.name}
                                icon={smartList.icon}
                                color={smartList.color}
                                count={smartListCounts[smartList.id as keyof typeof smartListCounts] || 0}
                                isSelected={selectedListId === smartList.id}
                                onClick={() => selectList(smartList.id)}
                                isSmartList
                            />
                        ))}
                    </div>
                </div>

                {/* User Lists - with Drag & Drop */}
                <div className="mb-4">
                    <h3 className="text-sm font-semibold text-purple-900 mb-3 px-2">
                        הרשימות שלי
                    </h3>

                    {lists.length === 0 ? (
                        <p className="text-sm text-purple-600 px-2 py-4 text-center">
                            אין רשימות עדיין
                        </p>
                    ) : (
                        <DndContext
                            sensors={sensors}
                            collisionDetection={closestCenter}
                            onDragEnd={handleDragEnd}
                        >
                            <SortableContext
                                items={lists.map(list => list.id)}
                                strategy={verticalListSortingStrategy}
                            >
                                <div className="space-y-1">
                                    {lists.map((list) => (
                                        <SortableListItem
                                            key={list.id}
                                            id={list.id}
                                            name={list.name}
                                            icon={list.icon}
                                            color={list.color}
                                            count={getListCount(list.id)}
                                            isSelected={selectedListId === list.id}
                                            onClick={() => selectList(list.id)}
                                            onDelete={() => handleDeleteList(list.id)}
                                        />
                                    ))}
                                </div>
                            </SortableContext>
                        </DndContext>
                    )}
                </div>
            </ScrollArea>

            {/* Export/Import Button */}
            <button
                onClick={() => window.open('/export-import-notes.html', '_blank')}
                className="
          mt-4 w-full py-3 px-4 rounded-xl
          bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700
          text-white font-medium
          flex items-center justify-center gap-2
          transition-all duration-200
          shadow-md hover:shadow-lg
          transform hover:scale-105
        "
                title="ייצוא ויבוא פתקים"
            >
                <Download className="w-4 h-4" />
                <Upload className="w-4 h-4" />
                <span>ייצוא / ייבוא</span>
            </button>

            {/* Create New List Button */}
            <button
                onClick={() => setIsCreateDialogOpen(true)}
                className="
          mt-4 w-full py-3 px-4 rounded-xl
          bg-purple-600 hover:bg-purple-700
          text-white font-medium
          flex items-center justify-center gap-2
          transition-all duration-200
          shadow-md hover:shadow-lg
          transform hover:scale-105
        "
            >
                <Plus className="w-5 h-5" />
                <span>רשימה חדשה</span>
            </button>

            {/* Create List Dialog */}
            <CreateListDialog
                open={isCreateDialogOpen}
                onClose={() => setIsCreateDialogOpen(false)}
            />
        </div>
    );
}