/**
 * Note Item Component
 * פריט פתק בודד
 */

'use client';

import { Note } from '@/types/notes';
import { useNotesStore } from '@/src/store/notesStore';
import { useNotesUIStore } from '@/src/store/notesUIStore';
import { formatDueDate, getDueDateStatus } from '@/src/lib/notes-helpers';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
    Star,
    Flag,
    Calendar,
    MoreVertical,
    Trash2,
    Copy,
    Edit,
    MoveRight,
    GripVertical,
} from 'lucide-react';

interface NoteItemProps {
    note: Note;
}

export default function NoteItem({ note }: NoteItemProps) {
    const { toggleDone, toggleImportant, toggleFlagged, duplicateNote, deleteNoteApi, updateNoteApi } = useNotesStore();
    const { openNoteEditor } = useNotesUIStore();

    const dueDateStatus = getDueDateStatus(note.dueDate);

    const handleToggleDone = async () => {
        toggleDone(note.id);
        await updateNoteApi(note.id, { done: !note.done });
    };

    const handleToggleImportant = async () => {
        toggleImportant(note.id);
        await updateNoteApi(note.id, { important: !note.important });
    };

    const handleToggleFlagged = async () => {
        toggleFlagged(note.id);
        await updateNoteApi(note.id, { flagged: !note.flagged });
    };

    const handleDelete = async () => {
        if (confirm('האם אתה בטוח שברצונך למחוק פתק זה?')) {
            await deleteNoteApi(note.id);
        }
    };

    const handleDuplicate = () => {
        duplicateNote(note.id);
    };

    return (
        <div
            className={`
        group p-4 rounded-xl border transition-all duration-200
        ${note.done ? 'bg-gray-50 border-gray-200' : 'bg-white border-gray-200 hover:border-purple-300 hover:shadow-md'}
        ${note.important && !note.done ? 'bg-yellow-50 border-yellow-200' : ''}
      `}
        >
            <div className="flex items-start gap-3">
                {/* Drag Handle */}
                <div className="mt-1 cursor-grab active:cursor-grabbing touch-none">
                    <GripVertical className="w-4 h-4 text-gray-400 hover:text-gray-600" />
                </div>

                {/* Checkbox */}
                <Checkbox
                    checked={note.done}
                    onCheckedChange={handleToggleDone}
                    className="mt-1"
                />

                {/* Content */}
                <div className="flex-1 min-w-0">
                    <div className="flex items-start justify-between gap-2">
                        <button
                            onClick={() => openNoteEditor(note.id)}
                            className="flex-1 text-right"
                        >
                            <h4
                                className={`
                  font-medium text-gray-900 mb-1
                  ${note.done ? 'line-through text-gray-500' : ''}
                `}
                            >
                                {note.title}
                            </h4>
                            {note.description && (
                                <p className="text-sm text-gray-600 line-clamp-2">
                                    {note.description}
                                </p>
                            )}
                        </button>

                        {/* Actions */}
                        <div className="flex items-center gap-1">
                            {/* Delete Button - Visible on hover */}
                            <button
                                onClick={handleDelete}
                                className="p-1.5 rounded-lg text-gray-400 hover:text-red-600 hover:bg-red-50 transition-colors opacity-0 group-hover:opacity-100"
                                title="מחק פתק"
                            >
                                <Trash2 className="w-4 h-4" />
                            </button>

                            {/* Important */}
                            <button
                                onClick={handleToggleImportant}
                                className={`p-1.5 rounded-lg transition-colors ${note.important
                                    ? 'text-yellow-500 bg-yellow-100'
                                    : 'text-gray-400 hover:text-yellow-500 hover:bg-yellow-50'
                                    }`}
                                title={note.important ? 'הסר מחשובים' : 'סמן כחשוב'}
                            >
                                <Star className={`w-4 h-4 ${note.important ? 'fill-current' : ''}`} />
                            </button>

                            {/* Flagged */}
                            <button
                                onClick={handleToggleFlagged}
                                className={`p-1.5 rounded-lg transition-colors ${note.flagged
                                    ? 'text-red-500 bg-red-100'
                                    : 'text-gray-400 hover:text-red-500 hover:bg-red-50'
                                    }`}
                                title={note.flagged ? 'הסר דגל' : 'סמן בדגל'}
                            >
                                <Flag className={`w-4 h-4 ${note.flagged ? 'fill-current' : ''}`} />
                            </button>

                            {/* More Menu */}
                            <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <button className="p-1.5 rounded-lg text-gray-400 hover:text-gray-600 hover:bg-gray-100 transition-colors">
                                        <MoreVertical className="w-4 h-4" />
                                    </button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                    <DropdownMenuItem onClick={() => openNoteEditor(note.id)}>
                                        <Edit className="w-4 h-4 ml-2" />
                                        <span>ערוך</span>
                                    </DropdownMenuItem>
                                    <DropdownMenuItem onClick={handleDuplicate}>
                                        <Copy className="w-4 h-4 ml-2" />
                                        <span>שכפל</span>
                                    </DropdownMenuItem>
                                    <DropdownMenuItem>
                                        <MoveRight className="w-4 h-4 ml-2" />
                                        <span>העבר לרשימה</span>
                                    </DropdownMenuItem>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem onClick={handleDelete} className="text-red-600">
                                        <Trash2 className="w-4 h-4 ml-2" />
                                        <span>מחק</span>
                                    </DropdownMenuItem>
                                </DropdownMenuContent>
                            </DropdownMenu>
                        </div>
                    </div>

                    {/* Footer */}
                    <div className="flex items-center gap-2 mt-2 flex-wrap">
                        {/* Due Date */}
                        {note.dueDate && (
                            <Badge
                                variant="outline"
                                className={`
                  ${dueDateStatus === 'overdue' ? 'border-red-500 text-red-700 bg-red-50' : ''}
                  ${dueDateStatus === 'today' ? 'border-blue-500 text-blue-700 bg-blue-50' : ''}
                  ${dueDateStatus === 'thisWeek' ? 'border-green-500 text-green-700 bg-green-50' : ''}
                `}
                            >
                                <Calendar className="w-3 h-3 ml-1" />
                                {formatDueDate(note.dueDate)}
                            </Badge>
                        )}

                        {/* Tags */}
                        {note.tags?.map((tag, index) => (
                            <Badge key={index} variant="secondary" className="bg-purple-100 text-purple-700">
                                {tag}
                            </Badge>
                        ))}

                        {/* Subtasks count */}
                        {note.subtasks && note.subtasks.length > 0 && (
                            <Badge variant="outline" className="text-gray-600">
                                {note.subtasks.filter(st => st.done).length}/{note.subtasks.length} משימות משנה
                            </Badge>
                        )}
                    </div>
                </div>
            </div>
        </div>
    );
}

