/**
 * Note Editor Component
 * עורך פתקים מתקדם (Sheet/Drawer)
 */

'use client';

import { useState, useEffect } from 'react';
import { useNotesStore } from '@/src/store/notesStore';
import { useNotesUIStore } from '@/src/store/notesUIStore';
import { useListsStore } from '@/src/store/listsStore';
import { Note, Subtask, Attachment } from '@/types/notes';
import {
    Sheet,
    SheetContent,
    SheetHeader,
    SheetTitle,
} from '@/components/ui/sheet';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import {
    Star,
    Flag,
    Calendar,
    List as ListIcon,
    Plus,
    X,
    Check,
} from 'lucide-react';
import toast from 'react-hot-toast';
import DateTimePicker from './DateTimePicker';
import FileUpload from './FileUpload';

export default function NoteEditor() {
    const { notes, updateNoteApi } = useNotesStore();
    const { lists } = useListsStore();
    const { isNoteEditorOpen, editingNoteId, closeNoteEditor } = useNotesUIStore();

    const currentNote = notes.find(n => n.id === editingNoteId);

    const [title, setTitle] = useState('');
    const [description, setDescription] = useState('');
    const [important, setImportant] = useState(false);
    const [flagged, setFlagged] = useState(false);
    const [dueDate, setDueDate] = useState('');
    const [reminderDate, setReminderDate] = useState<string | null>(null);
    const [reminderTime, setReminderTime] = useState<string | null>(null);
    const [reminderRecurrence, setReminderRecurrence] = useState<'none' | 'daily' | 'weekly' | 'monthly' | 'yearly'>('none');
    const [listId, setListId] = useState<string | null>(null);
    const [subtasks, setSubtasks] = useState<Subtask[]>([]);
    const [tags, setTags] = useState<string[]>([]);
    const [attachments, setAttachments] = useState<Attachment[]>([]);
    const [newSubtask, setNewSubtask] = useState('');
    const [newTag, setNewTag] = useState('');

    useEffect(() => {
        if (currentNote) {
            setTitle(currentNote.title);
            setDescription(currentNote.description || '');
            setImportant(currentNote.important);
            setFlagged(currentNote.flagged);
            setDueDate(currentNote.dueDate || '');
            setReminderDate(currentNote.reminderDate || null);
            setReminderTime(currentNote.reminderTime || null);
            setReminderRecurrence(currentNote.reminderRecurrence || 'none');
            setListId(currentNote.listId);
            setSubtasks(currentNote.subtasks || []);
            setTags(currentNote.tags || []);
            setAttachments(currentNote.attachments || []);
        }
    }, [currentNote]);

    const handleSave = async () => {
        if (!currentNote) return;

        if (!title.trim()) {
            toast.error('כותרת היא שדה חובה');
            return;
        }

        try {
            await updateNoteApi(currentNote.id, {
                title: title.trim(),
                description: description.trim(),
                important,
                flagged,
                dueDate: dueDate || null,
                reminderDate,
                reminderTime,
                reminderRecurrence,
                listId,
                subtasks,
                tags,
                attachments,
            });

            toast.success('הפתק עודכן בהצלחה! ✅');
            closeNoteEditor();
        } catch (error) {
            toast.error('שגיאה בעדכון הפתק');
            console.error('Error updating note:', error);
        }
    };

    const addSubtask = () => {
        if (!newSubtask.trim()) return;

        const subtask: Subtask = {
            id: Date.now().toString(),
            title: newSubtask.trim(),
            done: false,
        };

        setSubtasks([...subtasks, subtask]);
        setNewSubtask('');
    };

    const toggleSubtask = (id: string) => {
        setSubtasks(subtasks.map(st =>
            st.id === id ? { ...st, done: !st.done } : st
        ));
    };

    const removeSubtask = (id: string) => {
        setSubtasks(subtasks.filter(st => st.id !== id));
    };

    const addTag = () => {
        if (!newTag.trim() || tags.includes(newTag.trim())) return;

        setTags([...tags, newTag.trim()]);
        setNewTag('');
    };

    const removeTag = (tag: string) => {
        setTags(tags.filter(t => t !== tag));
    };

    if (!currentNote) return null;

    return (
        <Sheet open={isNoteEditorOpen} onOpenChange={closeNoteEditor}>
            <SheetContent side="left" className="w-full sm:w-[500px] overflow-y-auto" dir="rtl">
                <SheetHeader>
                    <SheetTitle>ערוך פתק</SheetTitle>
                </SheetHeader>

                <div className="mt-6 space-y-6">
                    {/* Title */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            כותרת *
                        </label>
                        <Input
                            value={title}
                            onChange={(e) => setTitle(e.target.value)}
                            placeholder="כותרת הפתק..."
                            className="text-lg"
                        />
                    </div>

                    {/* Description */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            תיאור
                        </label>
                        <Textarea
                            value={description}
                            onChange={(e) => setDescription(e.target.value)}
                            placeholder="הוסף תיאור מפורט..."
                            rows={4}
                        />
                    </div>

                    {/* Flags */}
                    <div className="flex gap-3">
                        <button
                            onClick={() => setImportant(!important)}
                            className={`
                flex-1 py-3 px-4 rounded-xl border-2 transition-all
                ${important
                                    ? 'border-yellow-500 bg-yellow-50 text-yellow-700'
                                    : 'border-gray-200 hover:border-yellow-300'
                                }
              `}
                        >
                            <Star className={`w-5 h-5 mx-auto mb-1 ${important ? 'fill-current' : ''}`} />
                            <span className="text-sm font-medium">חשוב</span>
                        </button>

                        <button
                            onClick={() => setFlagged(!flagged)}
                            className={`
                flex-1 py-3 px-4 rounded-xl border-2 transition-all
                ${flagged
                                    ? 'border-red-500 bg-red-50 text-red-700'
                                    : 'border-gray-200 hover:border-red-300'
                                }
              `}
                        >
                            <Flag className={`w-5 h-5 mx-auto mb-1 ${flagged ? 'fill-current' : ''}`} />
                            <span className="text-sm font-medium">דגל</span>
                        </button>
                    </div>

                    {/* Due Date */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            <Calendar className="w-4 h-4 inline ml-1" />
                            תאריך יעד
                        </label>
                        <Input
                            type="date"
                            value={dueDate}
                            onChange={(e) => setDueDate(e.target.value)}
                        />
                    </div>

                    {/* Reminder - הזכר לי */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            🔔 הזכר לי
                        </label>
                        <DateTimePicker
                            value={{ date: reminderDate, time: reminderTime, recurrence: reminderRecurrence }}
                            onChange={(date, time, recurrence) => {
                                setReminderDate(date);
                                setReminderTime(time);
                                setReminderRecurrence(recurrence || 'none');
                            }}
                        />
                    </div>

                    {/* File Attachments - הוסף קובץ */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            📎 קבצים מצורפים
                        </label>
                        <FileUpload
                            attachments={attachments}
                            onChange={setAttachments}
                        />
                    </div>

                    {/* List Selection */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            <ListIcon className="w-4 h-4 inline ml-1" />
                            רשימה
                        </label>
                        <select
                            value={listId || ''}
                            onChange={(e) => setListId(e.target.value || null)}
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500"
                            aria-label="בחר רשימה"
                        >
                            <option value="">ללא רשימה</option>
                            {lists.map(list => (
                                <option key={list.id} value={list.id}>
                                    {list.icon} {list.name}
                                </option>
                            ))}
                        </select>
                    </div>

                    {/* Subtasks */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            משימות משנה
                        </label>
                        <div className="space-y-2">
                            {subtasks.map(subtask => (
                                <div key={subtask.id} className="flex items-center gap-2 p-2 bg-gray-50 rounded-lg">
                                    <Checkbox
                                        checked={subtask.done}
                                        onCheckedChange={() => toggleSubtask(subtask.id)}
                                    />
                                    <span className={`flex-1 ${subtask.done ? 'line-through text-gray-500' : ''}`}>
                                        {subtask.title}
                                    </span>
                                    <button
                                        onClick={() => removeSubtask(subtask.id)}
                                        className="p-1 hover:bg-gray-200 rounded"
                                        aria-label="מחק משימת משנה"
                                    >
                                        <X className="w-4 h-4 text-gray-500" />
                                    </button>
                                </div>
                            ))}

                            <div className="flex gap-2">
                                <Input
                                    value={newSubtask}
                                    onChange={(e) => setNewSubtask(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && addSubtask()}
                                    placeholder="הוסף משימת משנה..."
                                />
                                <Button onClick={addSubtask} variant="outline" size="icon">
                                    <Plus className="w-4 h-4" />
                                </Button>
                            </div>
                        </div>
                    </div>

                    {/* Tags */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            תגיות
                        </label>
                        <div className="flex flex-wrap gap-2 mb-2">
                            {tags.map(tag => (
                                <Badge key={tag} variant="secondary" className="bg-purple-100 text-purple-700">
                                    {tag}
                                    <button
                                        onClick={() => removeTag(tag)}
                                        className="mr-1 hover:text-purple-900"
                                        aria-label="הסר תגית"
                                    >
                                        <X className="w-3 h-3" />
                                    </button>
                                </Badge>
                            ))}
                        </div>

                        <div className="flex gap-2">
                            <Input
                                value={newTag}
                                onChange={(e) => setNewTag(e.target.value)}
                                onKeyDown={(e) => e.key === 'Enter' && addTag()}
                                placeholder="הוסף תגית..."
                            />
                            <Button onClick={addTag} variant="outline" size="icon">
                                <Plus className="w-4 h-4" />
                            </Button>
                        </div>
                    </div>

                    {/* Actions */}
                    <div className="flex gap-3 pt-4">
                        <Button onClick={handleSave} className="flex-1 bg-purple-600 hover:bg-purple-700">
                            <Check className="w-4 h-4 ml-2" />
                            שמור שינויים
                        </Button>
                        <Button onClick={closeNoteEditor} variant="outline">
                            ביטול
                        </Button>
                    </div>
                </div>
            </SheetContent>
        </Sheet>
    );
}

