/**
 * Main Area Component
 * אזור תצוגת פתקים ראשי
 */

'use client';

import { useEffect, useMemo } from 'react';
import { useListsStore } from '@/src/store/listsStore';
import { useNotesStore } from '@/src/store/notesStore';
import { useNotesUIStore } from '@/src/store/notesUIStore';
import { smartLists, sortNotes, filterNotes, searchNotes } from '@/src/lib/notes-helpers';
import NotesList from './NotesList';
import AddNoteBar from './AddNoteBar';
import EmptyState from './EmptyState';
import SearchBar from './SearchBar';
import FiltersBar from './FiltersBar';

export default function MainArea() {
    const { selectedListId, lists } = useListsStore();
    const { notes, fetchNotes } = useNotesStore();
    const { filter, sortBy, searchQuery } = useNotesUIStore();

    // Get current list
    const currentList = useMemo(() => {
        if (!selectedListId) return null;

        const smartList = smartLists.find(l => l.id === selectedListId);
        if (smartList) return { ...smartList, isSmartList: true };

        const userList = lists.find(l => l.id === selectedListId);
        if (userList) return { ...userList, isSmartList: false };

        return null;
    }, [selectedListId, lists]);

    // Fetch notes when list changes
    useEffect(() => {
        if (selectedListId) {
            const isSmartList = smartLists.some(l => l.id === selectedListId);
            if (!isSmartList) {
                fetchNotes(selectedListId);
            }
        }
    }, [selectedListId, fetchNotes]);

    // Filter and sort notes
    const displayedNotes = useMemo(() => {
        let result = [...notes];

        // Smart list filtering
        if (selectedListId && currentList?.isSmartList) {
            const smartList = smartLists.find(l => l.id === selectedListId);
            if (smartList) {
                result = smartList.filter(result);
            }
        } else if (selectedListId) {
            // User list filtering
            result = result.filter(n => n.listId === selectedListId);
        }

        // Apply filter (all/open/completed)
        result = filterNotes(result, filter);

        // Apply search
        if (searchQuery) {
            result = searchNotes(result, searchQuery);
        }

        // Apply sorting
        result = sortNotes(result, sortBy);

        return result;
    }, [notes, selectedListId, currentList, filter, sortBy, searchQuery]);

    // No list selected
    if (!selectedListId || !currentList) {
        return (
            <EmptyState
                title="בחר רשימה כדי להתחיל"
                description="בחר רשימה מהתפריט הימני או צור רשימה חדשה"
            />
        );
    }

    const openNotesCount = displayedNotes.filter(n => !n.done).length;

    return (
        <div className="h-full flex flex-col bg-white">
            {/* Header */}
            <div className="border-b border-gray-200 p-6">
                <div className="flex items-center justify-between mb-4">
                    <div className="flex items-center gap-3">
                        {currentList.icon && (
                            <span className="text-3xl">{currentList.icon}</span>
                        )}
                        <div>
                            <h2 className="text-2xl font-bold text-gray-900">
                                {currentList.name}
                            </h2>
                            <p className="text-sm text-gray-500 mt-1">
                                {openNotesCount} משימות פתוחות
                            </p>
                        </div>
                    </div>

                    <SearchBar />
                </div>

                <FiltersBar />
            </div>

            {/* Notes List */}
            <div className="flex-1 overflow-y-auto">
                {displayedNotes.length === 0 ? (
                    <EmptyState
                        title={searchQuery ? 'לא נמצאו תוצאות' : 'אין פתקים ברשימה זו'}
                        description={searchQuery ? 'נסה חיפוש אחר' : 'הוסף פתק חדש כדי להתחיל'}
                    />
                ) : (
                    <NotesList notes={displayedNotes} />
                )}
            </div>

            {/* Add Note Bar */}
            <AddNoteBar listId={selectedListId} />
        </div>
    );
}

