/**
 * List Item Component
 * פריט רשימה בודד
 */

'use client';

import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Trash2, GripVertical } from 'lucide-react';
import {
    AlertDialog,
    AlertDialogAction,
    AlertDialogCancel,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogFooter,
    AlertDialogHeader,
    AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import { useState } from 'react';

interface ListItemProps {
    id: string;
    name: string;
    icon?: string | null;
    color?: string | null;
    count: number;
    isSelected: boolean;
    onClick: () => void;
    onDelete?: () => void;
    isSmartList?: boolean;
    isDraggable?: boolean;
    dragHandleProps?: any;
}

export default function ListItem({
    name,
    icon,
    color,
    count,
    isSelected,
    onClick,
    onDelete,
    isSmartList = false,
    isDraggable = false,
    dragHandleProps,
}: ListItemProps) {
    const [showDeleteDialog, setShowDeleteDialog] = useState(false);

    const handleDelete = (e: React.MouseEvent) => {
        e.stopPropagation();
        setShowDeleteDialog(true);
    };

    const confirmDelete = () => {
        onDelete?.();
        setShowDeleteDialog(false);
    };

    return (
        <>
            <div
                className={`
                    w-full px-4 py-3 rounded-xl
                    flex items-center justify-between gap-2
                    transition-all duration-200
                    ${isSelected
                        ? 'bg-purple-600 text-white shadow-md scale-105'
                        : 'bg-white/50 hover:bg-white text-purple-900 hover:shadow-sm'
                    }
                `}
            >
                {/* Drag Handle - רק ברשימות משתמש */}
                {isDraggable && !isSmartList && (
                    <div
                        {...dragHandleProps}
                        className="cursor-grab active:cursor-grabbing touch-none"
                    >
                        <GripVertical className="w-4 h-4 text-gray-400" />
                    </div>
                )}

                {/* Main Content */}
                <button
                    onClick={onClick}
                    className="flex-1 flex items-center justify-between gap-3"
                >
                    <div className="flex items-center gap-3">
                        {icon && (
                            <span className="text-xl">{icon}</span>
                        )}
                        {!icon && color && (
                            <div
                                className="w-3 h-3 rounded-full"
                                style={{ backgroundColor: color }}
                            />
                        )}
                        <span className="font-medium text-sm">{name}</span>
                    </div>

                    {count > 0 && (
                        <Badge
                            variant={isSelected ? 'secondary' : 'default'}
                            className={`
                                min-w-[24px] h-6 px-2 rounded-full
                                ${isSelected
                                    ? 'bg-white/20 text-white'
                                    : 'bg-purple-200 text-purple-900'
                                }
                            `}
                        >
                            {count}
                        </Badge>
                    )}
                </button>

                {/* Delete Button - רק ברשימות משתמש */}
                {!isSmartList && onDelete && (
                    <Button
                        variant="ghost"
                        size="icon"
                        onClick={handleDelete}
                        className={`h-8 w-8 ${isSelected ? 'text-white hover:bg-white/20' : 'text-red-500 hover:text-red-700 hover:bg-red-50'}`}
                    >
                        <Trash2 className="w-4 h-4" />
                    </Button>
                )}
            </div>

            {/* Delete Confirmation Dialog */}
            <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
                <AlertDialogContent dir="rtl">
                    <AlertDialogHeader>
                        <AlertDialogTitle>האם למחוק את הרשימה?</AlertDialogTitle>
                        <AlertDialogDescription>
                            פעולה זו תמחק את הרשימה "{name}" ואת כל הפתקים שבה.
                            <br />
                            לא ניתן לבטל פעולה זו.
                        </AlertDialogDescription>
                    </AlertDialogHeader>
                    <AlertDialogFooter className="gap-2">
                        <AlertDialogCancel>ביטול</AlertDialogCancel>
                        <AlertDialogAction
                            onClick={confirmDelete}
                            className="bg-red-600 hover:bg-red-700"
                        >
                            מחק
                        </AlertDialogAction>
                    </AlertDialogFooter>
                </AlertDialogContent>
            </AlertDialog>
        </>
    );
}

