/**
 * File Upload Component
 * קומפוננט להעלאת קבצים מצורפים
 */

'use client';

import React, { useRef, useState } from 'react';
import { Paperclip, X, FileText, Image as ImageIcon, File } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Attachment } from '@/types/notes';
import { v4 as uuidv4 } from 'uuid';

interface FileUploadProps {
    attachments: Attachment[];
    onChange: (attachments: Attachment[]) => void;
    maxFileSize?: number; // MB
    allowedTypes?: string[];
}

export default function FileUpload({
    attachments,
    onChange,
    maxFileSize = 10,
    allowedTypes = ['image/*', 'application/pdf', '.doc', '.docx', '.txt', '.xlsx', '.xls'],
}: FileUploadProps) {
    const fileInputRef = useRef<HTMLInputElement>(null);
    const [isUploading, setIsUploading] = useState(false);

    const handleFileSelect = async (e: React.ChangeEvent<HTMLInputElement>) => {
        const files = e.target.files;
        if (!files || files.length === 0) return;

        setIsUploading(true);

        try {
            const newAttachments: Attachment[] = [];

            for (let i = 0; i < files.length; i++) {
                const file = files[i];

                // Validate file size
                if (file.size > maxFileSize * 1024 * 1024) {
                    alert(`הקובץ ${file.name} גדול מדי. הגודל המקסימלי הוא ${maxFileSize}MB`);
                    continue;
                }

                // In a real app, upload to server/cloud storage
                // For now, we'll use data URL (for demo purposes)
                const reader = new FileReader();

                await new Promise<void>((resolve) => {
                    reader.onload = () => {
                        const attachment: Attachment = {
                            id: uuidv4(),
                            name: file.name,
                            type: file.type,
                            size: file.size,
                            url: reader.result as string, // In production, this would be a server URL
                            uploadedAt: new Date().toISOString(),
                        };
                        newAttachments.push(attachment);
                        resolve();
                    };
                    reader.readAsDataURL(file);
                });
            }

            onChange([...attachments, ...newAttachments]);
        } catch (error) {
            console.error('Error uploading files:', error);
            alert('שגיאה בהעלאת הקבצים');
        } finally {
            setIsUploading(false);
            if (fileInputRef.current) {
                fileInputRef.current.value = '';
            }
        }
    };

    const handleRemove = (attachmentId: string) => {
        onChange(attachments.filter(att => att.id !== attachmentId));
    };

    const getFileIcon = (type: string) => {
        if (type.startsWith('image/')) return <ImageIcon className="w-4 h-4" />;
        if (type === 'application/pdf') return <FileText className="w-4 h-4 text-red-500" />;
        return <File className="w-4 h-4" />;
    };

    const formatFileSize = (bytes: number) => {
        if (bytes < 1024) return bytes + ' B';
        if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
        return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
    };

    return (
        <div className="space-y-3">
            {/* Upload Button */}
            <div>
                <input
                    ref={fileInputRef}
                    type="file"
                    multiple
                    accept={allowedTypes.join(',')}
                    onChange={handleFileSelect}
                    className="hidden"
                    aria-label="העלה קבצים"
                />
                <Button
                    type="button"
                    variant="outline"
                    onClick={() => fileInputRef.current?.click()}
                    disabled={isUploading}
                    className="w-full justify-start text-right"
                >
                    <Paperclip className="w-4 h-4 ml-2" />
                    <span>{isUploading ? 'מעלה...' : 'הוסף קובץ'}</span>
                </Button>
            </div>

            {/* Attachments List */}
            {attachments.length > 0 && (
                <div className="space-y-2">
                    {attachments.map((attachment) => (
                        <div
                            key={attachment.id}
                            className="flex items-center justify-between p-3 bg-gray-50 rounded-lg border border-gray-200 hover:bg-gray-100 transition-colors"
                        >
                            <button
                                type="button"
                                onClick={() => handleRemove(attachment.id)}
                                className="p-1 hover:bg-red-100 rounded transition-colors"
                                aria-label="מחק קובץ"
                            >
                                <X className="w-4 h-4 text-red-500" />
                            </button>

                            <div className="flex-1 flex items-center justify-end gap-2">
                                <div className="text-right">
                                    <p className="text-sm font-medium text-gray-900 truncate max-w-[200px]">
                                        {attachment.name}
                                    </p>
                                    <p className="text-xs text-gray-500">
                                        {formatFileSize(attachment.size)}
                                    </p>
                                </div>
                                <div className="text-gray-500">
                                    {getFileIcon(attachment.type)}
                                </div>
                            </div>
                        </div>
                    ))}
                </div>
            )}
        </div>
    );
}

