/**
 * DateTime Picker Component for Reminders
 * בורר תאריך ושעה לתזכורות עם אופציות מהירות
 */

'use client';

import React, { useState } from 'react';
import { Calendar, Clock, X, Repeat } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { format, addDays, setHours, setMinutes, startOfDay } from 'date-fns';
import { he } from 'date-fns/locale';
import type { RecurrenceType } from '@/types/notes';

interface DateTimePickerProps {
    value?: { date: string | null; time: string | null; recurrence?: RecurrenceType };
    onChange: (date: string | null, time: string | null, recurrence?: RecurrenceType) => void;
    placeholder?: string;
}

export default function DateTimePicker({ value, onChange, placeholder }: DateTimePickerProps) {
    const [isOpen, setIsOpen] = useState(false);
    const [selectedDate, setSelectedDate] = useState<string | null>(value?.date || null);
    const [selectedTime, setSelectedTime] = useState<string | null>(value?.time || null);
    const [selectedRecurrence, setSelectedRecurrence] = useState<RecurrenceType>(value?.recurrence || 'none');

    const quickOptions = [
        {
            label: 'יום ה 4:00',
            icon: '☀️',
            getDateTime: () => {
                const today = new Date();
                const time = '16:00'; // 4:00 PM
                return { date: format(today, 'yyyy-MM-dd'), time };
            },
        },
        {
            label: 'יום ה 9:00',
            icon: '📅',
            getDateTime: () => {
                const today = new Date();
                const time = '09:00'; // 9:00 AM
                return { date: format(today, 'yyyy-MM-dd'), time };
            },
        },
        {
            label: 'מחר',
            icon: '📅',
            getDateTime: () => {
                const tomorrow = addDays(new Date(), 1);
                const time = '09:00';
                return { date: format(tomorrow, 'yyyy-MM-dd'), time };
            },
        },
        {
            label: 'בשבוע הבא',
            icon: '📅',
            getDateTime: () => {
                const nextWeek = addDays(new Date(), 7);
                const time = '09:00';
                return { date: format(nextWeek, 'yyyy-MM-dd'), time };
            },
        },
    ];

    const handleQuickOption = (option: typeof quickOptions[0]) => {
        const { date, time } = option.getDateTime();
        setSelectedDate(date);
        setSelectedTime(time);
        onChange(date, time, selectedRecurrence);
        // סגור אוטומטית אחרי בחירה מהירה
        setTimeout(() => setIsOpen(false), 300);
    };

    const handleCustomDate = (e: React.ChangeEvent<HTMLInputElement>) => {
        const date = e.target.value;
        setSelectedDate(date);
        onChange(date, selectedTime, selectedRecurrence);
        // סגור אוטומטית אחרי בחירת תאריך
        setTimeout(() => setIsOpen(false), 500);
    };

    const handleCustomTime = (e: React.ChangeEvent<HTMLInputElement>) => {
        const time = e.target.value;
        setSelectedTime(time);
        onChange(selectedDate, time, selectedRecurrence);
        // סגור אוטומטית אחרי בחירת שעה
        setTimeout(() => setIsOpen(false), 500);
    };

    const handleRecurrenceChange = (recurrence: RecurrenceType) => {
        setSelectedRecurrence(recurrence);
        onChange(selectedDate, selectedTime, recurrence);
    };

    const handleClear = () => {
        setSelectedDate(null);
        setSelectedTime(null);
        setSelectedRecurrence('none');
        onChange(null, null, 'none');
        setIsOpen(false);
    };

    const formatDisplay = () => {
        if (!selectedDate) return placeholder || 'הזכר לי';

        const dateObj = new Date(selectedDate);
        const formattedDate = format(dateObj, 'dd/MM/yyyy', { locale: he });

        let display = formattedDate;
        if (selectedTime) {
            display += ` ${selectedTime}`;
        }

        // הוסף אינדיקטור לתזכורת חוזרת
        if (selectedRecurrence && selectedRecurrence !== 'none') {
            const recurrenceLabels = {
                daily: '🔄 יומי',
                weekly: '🔄 שבועי',
                monthly: '🔄 חודשי',
                yearly: '🔄 שנתי',
            };
            display += ` (${recurrenceLabels[selectedRecurrence]})`;
        }

        return display;
    };

    return (
        <div className="relative">
            <Button
                type="button"
                variant="outline"
                className="w-full justify-start text-right"
                onClick={() => setIsOpen(!isOpen)}
            >
                <Clock className="w-4 h-4 ml-2" />
                <span className={selectedDate ? 'text-gray-900' : 'text-gray-500'}>
                    {formatDisplay()}
                </span>
            </Button>

            {isOpen && (
                <div className="absolute left-0 top-full mt-2 w-80 bg-white rounded-lg shadow-xl border border-gray-200 z-50 p-4">
                    {/* Quick Options */}
                    <div className="space-y-2 mb-4">
                        <h4 className="text-sm font-semibold text-gray-700 mb-2">אופציות מהירות</h4>
                        {quickOptions.map((option, index) => (
                            <button
                                key={index}
                                type="button"
                                onClick={() => handleQuickOption(option)}
                                className="w-full flex items-center justify-start px-3 py-2 rounded-lg hover:bg-blue-50 transition-colors text-right border border-transparent hover:border-blue-200"
                            >
                                <span className="text-xl ml-3">{option.icon}</span>
                                <span className="text-sm font-medium text-gray-700">{option.label}</span>
                            </button>
                        ))}
                    </div>

                    <div className="border-t pt-4">
                        <h4 className="text-sm font-semibold text-gray-700 mb-3">בחר מועד מותאם אישית</h4>

                        {/* Date Picker */}
                        <div className="mb-3">
                            <label className="block text-sm text-gray-600 mb-1 text-right">תאריך</label>
                            <Input
                                type="date"
                                value={selectedDate || ''}
                                onChange={handleCustomDate}
                                className="w-full text-right"
                            />
                        </div>

                        {/* Time Picker */}
                        <div className="mb-4">
                            <label className="block text-sm text-gray-600 mb-1 text-right">שעה</label>
                            <Input
                                type="time"
                                value={selectedTime || ''}
                                onChange={handleCustomTime}
                                className="w-full text-right"
                            />
                        </div>

                        {/* Recurrence Options */}
                        <div className="mb-4">
                            <label className="block text-sm font-semibold text-gray-700 mb-2 text-right flex items-center justify-end">
                                <Repeat className="w-4 h-4 mr-1" />
                                חזרה על עצמו
                            </label>
                            <div className="grid grid-cols-2 gap-2">
                                <button
                                    type="button"
                                    onClick={() => handleRecurrenceChange('none')}
                                    className={`px-3 py-2 rounded-lg text-sm font-medium transition-all ${selectedRecurrence === 'none'
                                            ? 'bg-gray-700 text-white'
                                            : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                        }`}
                                >
                                    ללא
                                </button>
                                <button
                                    type="button"
                                    onClick={() => handleRecurrenceChange('daily')}
                                    className={`px-3 py-2 rounded-lg text-sm font-medium transition-all ${selectedRecurrence === 'daily'
                                            ? 'bg-blue-600 text-white'
                                            : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                        }`}
                                >
                                    🔄 כל יום
                                </button>
                                <button
                                    type="button"
                                    onClick={() => handleRecurrenceChange('weekly')}
                                    className={`px-3 py-2 rounded-lg text-sm font-medium transition-all ${selectedRecurrence === 'weekly'
                                            ? 'bg-green-600 text-white'
                                            : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                        }`}
                                >
                                    🔄 שבועי
                                </button>
                                <button
                                    type="button"
                                    onClick={() => handleRecurrenceChange('monthly')}
                                    className={`px-3 py-2 rounded-lg text-sm font-medium transition-all ${selectedRecurrence === 'monthly'
                                            ? 'bg-purple-600 text-white'
                                            : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                        }`}
                                >
                                    🔄 חודשי
                                </button>
                                <button
                                    type="button"
                                    onClick={() => handleRecurrenceChange('yearly')}
                                    className={`px-3 py-2 rounded-lg text-sm font-medium transition-all ${selectedRecurrence === 'yearly'
                                            ? 'bg-orange-600 text-white'
                                            : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                        }`}
                                >
                                    🔄 שנתי
                                </button>
                            </div>
                        </div>

                        {/* Actions */}
                        <div className="flex gap-2">
                            <Button
                                type="button"
                                variant="outline"
                                onClick={handleClear}
                                className="flex-1"
                            >
                                <X className="w-4 h-4 ml-1" />
                                נקה
                            </Button>
                            <Button
                                type="button"
                                onClick={() => setIsOpen(false)}
                                className="flex-1 bg-blue-600 hover:bg-blue-700"
                            >
                                אישור
                            </Button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}

