/**
 * Create List Dialog Component
 * דיאלוג יצירת רשימה חדשה
 */

'use client';

import { useState } from 'react';
import { useListsStore } from '@/src/store/listsStore';
import {
    Dialog,
    DialogContent,
    DialogHeader,
    DialogTitle,
} from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import toast from 'react-hot-toast';

interface CreateListDialogProps {
    open: boolean;
    onClose: () => void;
}

const emojiOptions = ['📝', '💼', '🏠', '🎯', '💰', '🛒', '✈️', '🎨', '📚', '🏋️', '🎵', '🌙'];
const colorOptions = [
    '#4A90E2', '#27AE60', '#F39C12', '#E74C3C', '#9B59B6',
    '#1ABC9C', '#34495E', '#E67E22', '#95A5A6', '#16A085',
];

export default function CreateListDialog({ open, onClose }: CreateListDialogProps) {
    const { createList } = useListsStore();
    const [name, setName] = useState('');
    const [selectedEmoji, setSelectedEmoji] = useState('📝');
    const [selectedColor, setSelectedColor] = useState('#4A90E2');

    const handleSubmit = async () => {
        if (!name.trim()) {
            toast.error('נא להזין שם לרשימה');
            return;
        }

        try {
            await createList({
                name: name.trim(),
                icon: selectedEmoji,
                color: selectedColor,
                sortOrder: 0,
            });

            toast.success('הרשימה נוצרה בהצלחה! ✅');
            onClose();
            setName('');
            setSelectedEmoji('📝');
            setSelectedColor('#4A90E2');
        } catch (error) {
            toast.error('שגיאה ביצירת הרשימה');
            console.error('Error creating list:', error);
        }
    };

    return (
        <Dialog open={open} onOpenChange={onClose}>
            <DialogContent dir="rtl">
                <DialogHeader>
                    <DialogTitle>רשימה חדשה</DialogTitle>
                </DialogHeader>

                <div className="space-y-6 mt-4">
                    {/* Name */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            שם הרשימה *
                        </label>
                        <Input
                            value={name}
                            onChange={(e) => setName(e.target.value)}
                            placeholder="לדוגמה: משימות עבודה"
                            autoFocus
                        />
                    </div>

                    {/* Emoji */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            אייקון
                        </label>
                        <div className="grid grid-cols-6 gap-2">
                            {emojiOptions.map((emoji) => (
                                <button
                                    key={emoji}
                                    onClick={() => setSelectedEmoji(emoji)}
                                    className={`
                    p-3 rounded-lg text-2xl transition-all
                    ${selectedEmoji === emoji
                                            ? 'bg-purple-100 ring-2 ring-purple-500 scale-110'
                                            : 'bg-gray-100 hover:bg-gray-200'
                                        }
                  `}
                                >
                                    {emoji}
                                </button>
                            ))}
                        </div>
                    </div>

                    {/* Color */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                            צבע
                        </label>
                        <div className="grid grid-cols-5 gap-2">
                            {colorOptions.map((color) => (
                                <button
                                    key={color}
                                    onClick={() => setSelectedColor(color)}
                                    className={`
                    w-12 h-12 rounded-lg transition-all
                    ${selectedColor === color
                                            ? 'ring-4 ring-gray-400 scale-110'
                                            : 'hover:scale-105'
                                        }
                  `}
                                    style={{ backgroundColor: color }}
                                />
                            ))}
                        </div>
                    </div>

                    {/* Actions */}
                    <div className="flex gap-3">
                        <Button onClick={handleSubmit} className="flex-1 bg-purple-600 hover:bg-purple-700">
                            צור רשימה
                        </Button>
                        <Button onClick={onClose} variant="outline">
                            ביטול
                        </Button>
                    </div>
                </div>
            </DialogContent>
        </Dialog>
    );
}



