'use client';

import React, { useState } from 'react';
import { Calendar, Plus, X, CheckCircle } from 'lucide-react';

interface EventTaskLinkProps {
    taskId?: number;
    eventId?: number;
    eventTitle?: string;
    eventStart?: string;
    onLink?: (eventId: number) => void;
    onUnlink?: () => void;
}

export default function EventTaskLink({
    taskId,
    eventId,
    eventTitle,
    eventStart,
    onLink,
    onUnlink
}: EventTaskLinkProps) {
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const handleLink = async (eventId: number) => {
        if (!taskId) return;

        setLoading(true);
        setError(null);
        try {
            const response = await fetch(`/api/tasks/${taskId}/link-event`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ event_id: eventId })
            });

            if (response.ok) {
                if (onLink) onLink(eventId);
            } else {
                const errorData = await response.json();
                setError(errorData.error || 'שגיאה בקישור לאירוע');
            }
        } catch (err) {
            setError('שגיאה בקישור לאירוע');
        } finally {
            setLoading(false);
        }
    };

    const handleUnlink = async () => {
        if (!taskId) return;

        setLoading(true);
        setError(null);
        try {
            const response = await fetch(`/api/tasks/${taskId}/link-event`, {
                method: 'DELETE'
            });

            if (response.ok) {
                if (onUnlink) onUnlink();
            } else {
                const errorData = await response.json();
                setError(errorData.error || 'שגיאה בניתוק מאירוע');
            }
        } catch (err) {
            setError('שגיאה בניתוק מאירוע');
        } finally {
            setLoading(false);
        }
    };

    const formatDateTime = (datetime: string) => {
        const date = new Date(datetime);
        return {
            date: date.toLocaleDateString('he-IL'),
            time: date.toLocaleTimeString('he-IL', { hour: '2-digit', minute: '2-digit' })
        };
    };

    if (eventId && eventTitle) {
        const startTime = eventStart ? formatDateTime(eventStart) : null;

        return (
            <div className="flex items-center gap-2">
                <div className="flex items-center gap-2 px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm">
                    <Calendar className="w-4 h-4" />
                    <span>קשור לאירוע</span>
                </div>
                <div className="text-sm text-gray-600">
                    {eventTitle}
                    {startTime && (
                        <span className="text-gray-400 mr-2">
                            ({startTime.date} {startTime.time})
                        </span>
                    )}
                </div>
                <button
                    onClick={handleUnlink}
                    disabled={loading}
                    className="p-1 text-gray-400 hover:text-red-600 transition-colors disabled:opacity-50"
                    title="נתק מאירוע"
                >
                    <X className="w-4 h-4" />
                </button>
                {error && (
                    <div className="text-xs text-red-600">{error}</div>
                )}
            </div>
        );
    }

    return (
        <div className="flex items-center gap-2">
            <button
                onClick={() => {/* TODO: Open event selection modal */ }}
                disabled={loading}
                className="flex items-center gap-2 px-3 py-1 border border-gray-300 rounded-md hover:bg-gray-50 transition-colors disabled:opacity-50"
                title="קשר לאירוע"
            >
                <Plus className="w-4 h-4" />
                <span>קשר לאירוע</span>
            </button>
            {error && (
                <div className="text-xs text-red-600">{error}</div>
            )}
        </div>
    );
}
