'use client';

import React, { useEffect, useState } from 'react';
import { Calendar, Clock, MapPin, Users, Edit, Trash2, Eye, Plus } from 'lucide-react';

interface Event {
    id: number;
    title: string;
    description?: string;
    start_datetime: string;
    end_datetime?: string;
    location?: string;
    event_type: string;
    status: string;
    customer_name?: string;
    customer_company?: string;
    domain_name?: string;
    project_title?: string;
    attendees?: string;
}

export default function EventList() {
    const [events, setEvents] = useState<Event[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [searchTerm, setSearchTerm] = useState('');
    const [statusFilter, setStatusFilter] = useState('all');
    const [typeFilter, setTypeFilter] = useState('all');
    const [currentPage, setCurrentPage] = useState(1);
    const [totalPages, setTotalPages] = useState(1);
    const pageSize = 10;

    // Event type options
    const eventTypeOptions = [
        { value: 'all', label: 'כל הסוגים' },
        { value: 'meeting', label: 'פגישה' },
        { value: 'call', label: 'שיחה' },
        { value: 'presentation', label: 'מצגת' },
        { value: 'webinar', label: 'וובינר' },
        { value: 'deadline', label: 'דדליין' },
        { value: 'reminder', label: 'תזכורת' },
        { value: 'other', label: 'אחר' }
    ];

    // Status options
    const statusOptions = [
        { value: 'all', label: 'כל הסטטוסים' },
        { value: 'scheduled', label: 'מתוכנן' },
        { value: 'in_progress', label: 'בתהליך' },
        { value: 'completed', label: 'הושלם' },
        { value: 'cancelled', label: 'בוטל' },
        { value: 'postponed', label: 'נדחה' }
    ];

    // Event type icons
    const getEventTypeIcon = (type: string) => {
        const icons: Record<string, string> = {
            meeting: '🤝',
            call: '📞',
            presentation: '📊',
            webinar: '💻',
            deadline: '⏰',
            reminder: '🔔',
            other: '📌'
        };
        return icons[type] || '📌';
    };

    // Status colors
    const getStatusColor = (status: string) => {
        const colors: Record<string, string> = {
            scheduled: 'bg-blue-100 text-blue-800',
            in_progress: 'bg-yellow-100 text-yellow-800',
            completed: 'bg-green-100 text-green-800',
            cancelled: 'bg-red-100 text-red-800',
            postponed: 'bg-gray-100 text-gray-800'
        };
        return colors[status] || 'bg-gray-100 text-gray-800';
    };

    // Status labels
    const getStatusLabel = (status: string) => {
        const labels: Record<string, string> = {
            scheduled: 'מתוכנן',
            in_progress: 'בתהליך',
            completed: 'הושלם',
            cancelled: 'בוטל',
            postponed: 'נדחה'
        };
        return labels[status] || status;
    };

    // Format datetime
    const formatDateTime = (datetime: string) => {
        const date = new Date(datetime);
        return {
            date: date.toLocaleDateString('he-IL'),
            time: date.toLocaleTimeString('he-IL', { hour: '2-digit', minute: '2-digit' })
        };
    };

    // Load events
    const loadEvents = async () => {
        setLoading(true);
        setError(null);
        try {
            const offset = (currentPage - 1) * pageSize;
            const params = new URLSearchParams({
                limit: pageSize.toString(),
                offset: offset.toString()
            });

            if (searchTerm) params.append('q', searchTerm);
            if (statusFilter !== 'all') params.append('status', statusFilter);
            if (typeFilter !== 'all') params.append('event_type', typeFilter);

            const url = `/api/events?${params}`;
            console.log('🔄 Loading events from:', url);

            const response = await fetch(url);
            console.log('📡 Response status:', response.status);
            console.log('📡 Response URL:', response.url);

            if (response.ok) {
                const data = await response.json();
                console.log('✅ Success data:', data);
                console.log('📊 Events count:', data.events?.length || 0);
                console.log('📊 Total count:', data.total || 0);
                setEvents(data.events || []);
                setTotalPages(Math.ceil((data.total || 0) / pageSize));
            } else {
                const text = await response.text();
                console.error('❌ Error response text:', text);
                console.error('❌ Response status:', response.status);
                try {
                    const errorData = JSON.parse(text);
                    setError(errorData.error || 'שגיאה בטעינת האירועים');
                    console.error('❌ API Error:', errorData);
                } catch {
                    setError(`שגיאה ${response.status}: ${text}`);
                }
            }
        } catch (err) {
            console.error('Fetch error:', err);
            setError('שגיאה בטעינת האירועים');
        } finally {
            setLoading(false);
        }
    };

    // Delete event
    const deleteEvent = async (id: number) => {
        if (!confirm('האם אתה בטוח שברצונך למחוק את האירוע?')) return;

        try {
            const response = await fetch(`/api/events/${id}`, { method: 'DELETE' });
            if (response.ok) {
                loadEvents();
            } else {
                setError('שגיאה במחיקת האירוע');
            }
        } catch (err) {
            setError('שגיאה במחיקת האירוע');
        }
    };

    // Load events on mount and when filters change
    useEffect(() => {
        loadEvents();
    }, [currentPage, searchTerm, statusFilter, typeFilter]);

    // Filtered events for display
    const filteredEvents = events.filter(event => {
        const matchesSearch = !searchTerm ||
            event.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
            event.description?.toLowerCase().includes(searchTerm.toLowerCase()) ||
            event.customer_name?.toLowerCase().includes(searchTerm.toLowerCase());

        return matchesSearch;
    });

    return (
        <div className="space-y-6">
            {/* Filters */}
            <div className="bg-white p-4 rounded-lg shadow">
                <div className="flex flex-col md:flex-row gap-4">
                    <div className="flex-1">
                        <input
                            type="text"
                            placeholder="חיפוש אירועים..."
                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            value={searchTerm}
                            onChange={e => setSearchTerm(e.target.value)}
                            dir="rtl"
                        />
                    </div>
                    <select
                        className="border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        value={statusFilter}
                        onChange={e => setStatusFilter(e.target.value)}
                        aria-label="סינון לפי סטטוס"
                    >
                        {statusOptions.map(option => (
                            <option key={option.value} value={option.value}>
                                {option.label}
                            </option>
                        ))}
                    </select>
                    <select
                        className="border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        value={typeFilter}
                        onChange={e => setTypeFilter(e.target.value)}
                        aria-label="סינון לפי סוג"
                    >
                        {eventTypeOptions.map(option => (
                            <option key={option.value} value={option.value}>
                                {option.label}
                            </option>
                        ))}
                    </select>
                </div>
            </div>

            {/* Events List */}
            <div className="bg-white shadow rounded-lg overflow-hidden">
                {loading && (
                    <div className="p-8 text-center text-gray-500">
                        טוען אירועים...
                    </div>
                )}

                {error && !loading && (
                    <div className="p-8 text-center text-red-600">
                        {error}
                    </div>
                )}

                {!loading && !error && filteredEvents.length === 0 && (
                    <div className="p-8 text-center text-gray-500">
                        אין אירועים
                    </div>
                )}

                {!loading && !error && filteredEvents.length > 0 && (
                    <div className="divide-y divide-gray-200">
                        {filteredEvents.map(event => {
                            const startTime = formatDateTime(event.start_datetime);
                            const endTime = event.end_datetime ? formatDateTime(event.end_datetime) : null;
                            const attendees = event.attendees ? JSON.parse(event.attendees) : [];

                            return (
                                <div key={event.id} className="p-6 hover:bg-gray-50">
                                    <div className="flex items-start justify-between">
                                        <div className="flex-1">
                                            <div className="flex items-center gap-3 mb-2">
                                                <span className="text-2xl">
                                                    {getEventTypeIcon(event.event_type)}
                                                </span>
                                                <h3 className="text-lg font-medium text-gray-900">
                                                    {event.title}
                                                </h3>
                                                <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs ${getStatusColor(event.status)}`}>
                                                    {getStatusLabel(event.status)}
                                                </span>
                                            </div>

                                            {event.description && (
                                                <p className="text-gray-600 mb-3">{event.description}</p>
                                            )}

                                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm text-gray-600">
                                                <div className="flex items-center gap-2">
                                                    <Calendar className="w-4 h-4" />
                                                    <span>{startTime.date}</span>
                                                </div>
                                                <div className="flex items-center gap-2">
                                                    <Clock className="w-4 h-4" />
                                                    <span>
                                                        {startTime.time}
                                                        {endTime && ` - ${endTime.time}`}
                                                    </span>
                                                </div>
                                                {event.location && (
                                                    <div className="flex items-center gap-2">
                                                        <MapPin className="w-4 h-4" />
                                                        <span>{event.location}</span>
                                                    </div>
                                                )}
                                                {attendees.length > 0 && (
                                                    <div className="flex items-center gap-2">
                                                        <Users className="w-4 h-4" />
                                                        <span>{attendees.length} משתתפים</span>
                                                    </div>
                                                )}
                                            </div>

                                            {(event.customer_name || event.domain_name || event.project_title) && (
                                                <div className="mt-3 text-sm text-gray-500">
                                                    {event.customer_name && (
                                                        <span className="inline-block bg-blue-100 text-blue-800 px-2 py-1 rounded text-xs mr-2">
                                                            👤 {event.customer_name}
                                                        </span>
                                                    )}
                                                    {event.domain_name && (
                                                        <span className="inline-block bg-green-100 text-green-800 px-2 py-1 rounded text-xs mr-2">
                                                            🌐 {event.domain_name}
                                                        </span>
                                                    )}
                                                    {event.project_title && (
                                                        <span className="inline-block bg-purple-100 text-purple-800 px-2 py-1 rounded text-xs mr-2">
                                                            📊 {event.project_title}
                                                        </span>
                                                    )}
                                                </div>
                                            )}
                                        </div>

                                        <div className="flex items-center gap-2">
                                            <a
                                                href={`/events/${event.id}`}
                                                className="p-2 text-gray-400 hover:text-blue-600 transition-colors"
                                                title="צפה באירוע"
                                            >
                                                <Eye className="w-4 h-4" />
                                            </a>
                                            <a
                                                href={`/events/${event.id}/edit`}
                                                className="p-2 text-gray-400 hover:text-yellow-600 transition-colors"
                                                title="ערוך אירוע"
                                            >
                                                <Edit className="w-4 h-4" />
                                            </a>
                                            <button
                                                onClick={() => deleteEvent(event.id)}
                                                className="p-2 text-gray-400 hover:text-red-600 transition-colors"
                                                title="מחק אירוע"
                                            >
                                                <Trash2 className="w-4 h-4" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            );
                        })}
                    </div>
                )}
            </div>

            {/* Pagination */}
            {totalPages > 1 && (
                <div className="flex items-center justify-between">
                    <div className="text-sm text-gray-600">
                        עמוד {currentPage} מתוך {totalPages}
                    </div>
                    <div className="flex items-center gap-2">
                        <button
                            onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                            disabled={currentPage === 1}
                            className="px-3 py-1 border border-gray-300 rounded disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
                        >
                            קודם
                        </button>
                        <button
                            onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
                            disabled={currentPage === totalPages}
                            className="px-3 py-1 border border-gray-300 rounded disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
                        >
                            הבא
                        </button>
                    </div>
                </div>
            )}
        </div>
    );
}
