'use client';

import { useState, useEffect } from 'react';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';

interface Domain {
    id: string;
    domain_name: string;
    customer_id: string;
}

interface DomainSelectProps {
    customerId?: string;
    value?: string;
    onChange: (value: string | undefined) => void;
    placeholder?: string;
}

export function DomainSelect({
    customerId,
    value,
    onChange,
    placeholder = "בחר דומיין"
}: DomainSelectProps) {
    const [domains, setDomains] = useState<Domain[]>([]);
    const [loading, setLoading] = useState(false);

    useEffect(() => {
        if (!customerId) {
            setDomains([]);
            return;
        }

        const fetchDomains = async () => {
            setLoading(true);
            try {
                const response = await fetch(`/api/domains?customer_id=${customerId}`);
                if (response.ok) {
                    const data = await response.json();
                    setDomains(data.domains || []);
                }
            } catch (error) {
                console.error('Error fetching domains:', error);
            } finally {
                setLoading(false);
            }
        };

        fetchDomains();
    }, [customerId]);

    if (!customerId) {
        return (
            <Select disabled>
                <SelectTrigger>
                    <SelectValue placeholder="בחר קודם לקוח" />
                </SelectTrigger>
            </Select>
        );
    }

    if (loading) {
        return (
            <Select disabled>
                <SelectTrigger>
                    <SelectValue placeholder="טוען דומיינים..." />
                </SelectTrigger>
            </Select>
        );
    }

    return (
        <Select
            value={value}
            onValueChange={onChange}
        >
            <SelectTrigger>
                <SelectValue placeholder={placeholder} />
            </SelectTrigger>
            <SelectContent>
                {domains.map((domain) => (
                    <SelectItem key={domain.id} value={domain.id}>
                        {domain.domain_name}
                    </SelectItem>
                ))}
            </SelectContent>
        </Select>
    );
}
