'use client';

import React, { useEffect, useState } from 'react';
import { Calendar, CheckCircle, Clock, AlertCircle, Plus, Eye, Edit, Trash2 } from 'lucide-react';

interface Event {
    id: number;
    title: string;
    start_datetime: string;
    end_datetime?: string;
    event_type: string;
    status: string;
    customer_name?: string;
    location?: string;
    type: 'event';
}

interface Task {
    id: number;
    title: string;
    due_date?: string;
    status: string;
    priority: string;
    progress: number;
    customer_name?: string;
    event_title?: string;
    event_start?: string;
    type: 'task';
}

interface IntegratedItem extends Event, Task {
    type: 'event' | 'task';
}

export default function IntegratedDashboard() {
    const [items, setItems] = useState<IntegratedItem[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [view, setView] = useState<'today' | 'week' | 'month'>('today');
    const [filter, setFilter] = useState<'all' | 'events' | 'tasks' | 'linked'>('all');

    // Load integrated data
    const loadData = async () => {
        setLoading(true);
        try {
            const today = new Date().toISOString().split('T')[0];
            const params = new URLSearchParams({
                date: today,
                view: view
            });

            const response = await fetch(`/api/calendar?${params}`);
            if (response.ok) {
                const data = await response.json();
                const integratedItems: IntegratedItem[] = [
                    ...(data.events || []).map((event: any) => ({
                        ...event,
                        type: 'event' as const
                    })),
                    ...(data.tasks || []).map((task: any) => ({
                        ...task,
                        type: 'task' as const
                    }))
                ];
                setItems(integratedItems);
            } else {
                setError('שגיאה בטעינת הנתונים');
            }
        } catch (err) {
            setError('שגיאה בטעינת הנתונים');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        loadData();
    }, [view]);

    // Filter items based on current filter
    const filteredItems = items.filter(item => {
        switch (filter) {
            case 'events':
                return item.type === 'event';
            case 'tasks':
                return item.type === 'task';
            case 'linked':
                return item.type === 'task' && item.event_title;
            default:
                return true;
        }
    });

    // Sort items by datetime
    const sortedItems = filteredItems.sort((a, b) => {
        const aTime = a.type === 'event' ? a.start_datetime : a.due_date;
        const bTime = b.type === 'event' ? b.start_datetime : b.due_date;
        return new Date(aTime || '').getTime() - new Date(bTime || '').getTime();
    });

    // Get event type icon
    const getEventTypeIcon = (type: string) => {
        const icons: Record<string, string> = {
            meeting: '🤝',
            call: '📞',
            presentation: '📊',
            webinar: '💻',
            deadline: '⏰',
            reminder: '🔔',
            other: '📌'
        };
        return icons[type] || '📌';
    };

    // Get status color
    const getStatusColor = (status: string, type: 'event' | 'task') => {
        if (type === 'event') {
            const colors: Record<string, string> = {
                scheduled: 'bg-blue-100 text-blue-800',
                in_progress: 'bg-yellow-100 text-yellow-800',
                completed: 'bg-green-100 text-green-800',
                cancelled: 'bg-red-100 text-red-800',
                postponed: 'bg-gray-100 text-gray-800'
            };
            return colors[status] || 'bg-gray-100 text-gray-800';
        } else {
            const colors: Record<string, string> = {
                pending: 'bg-yellow-100 text-yellow-800',
                in_progress: 'bg-blue-100 text-blue-800',
                completed: 'bg-green-100 text-green-800',
                cancelled: 'bg-red-100 text-red-800',
                on_hold: 'bg-purple-100 text-purple-800'
            };
            return colors[status] || 'bg-gray-100 text-gray-800';
        }
    };

    // Get status label
    const getStatusLabel = (status: string, type: 'event' | 'task') => {
        if (type === 'event') {
            const labels: Record<string, string> = {
                scheduled: 'מתוכנן',
                in_progress: 'בתהליך',
                completed: 'הושלם',
                cancelled: 'בוטל',
                postponed: 'נדחה'
            };
            return labels[status] || status;
        } else {
            const labels: Record<string, string> = {
                pending: 'ממתין',
                in_progress: 'בתהליך',
                completed: 'הושלם',
                cancelled: 'בוטל',
                on_hold: 'בהמתנה'
            };
            return labels[status] || status;
        }
    };

    // Format datetime
    const formatDateTime = (datetime: string) => {
        const date = new Date(datetime);
        return {
            date: date.toLocaleDateString('he-IL'),
            time: date.toLocaleTimeString('he-IL', { hour: '2-digit', minute: '2-digit' })
        };
    };

    // Get priority color for tasks
    const getPriorityColor = (priority: string) => {
        const colors: Record<string, string> = {
            urgent: 'bg-red-100 text-red-800',
            high: 'bg-orange-100 text-orange-800',
            medium: 'bg-yellow-100 text-yellow-800',
            low: 'bg-green-100 text-green-800'
        };
        return colors[priority] || 'bg-gray-100 text-gray-800';
    };

    // Get priority label
    const getPriorityLabel = (priority: string) => {
        const labels: Record<string, string> = {
            urgent: 'דחוף',
            high: 'גבוה',
            medium: 'בינוני',
            low: 'נמוך'
        };
        return labels[priority] || priority;
    };

    if (loading) {
        return (
            <div className="space-y-6">
                <div className="bg-white p-6 rounded-lg shadow">
                    <div className="animate-pulse">
                        <div className="h-6 bg-gray-200 rounded w-1/4 mb-4"></div>
                        <div className="space-y-3">
                            {[1, 2, 3].map(i => (
                                <div key={i} className="h-16 bg-gray-200 rounded"></div>
                            ))}
                        </div>
                    </div>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded">
                {error}
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header with filters */}
            <div className="bg-white p-6 rounded-lg shadow">
                <div className="flex items-center justify-between mb-4">
                    <h2 className="text-xl font-semibold">דשבורד משולב</h2>
                    <div className="flex items-center gap-4">
                        <select
                            value={view}
                            onChange={e => setView(e.target.value as any)}
                            className="border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            aria-label="בחירת תצוגה"
                        >
                            <option value="today">היום</option>
                            <option value="week">השבוע</option>
                            <option value="month">החודש</option>
                        </select>
                        <select
                            value={filter}
                            onChange={e => setFilter(e.target.value as any)}
                            className="border border-gray-300 rounded-md px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            aria-label="סינון"
                        >
                            <option value="all">הכל</option>
                            <option value="events">אירועים</option>
                            <option value="tasks">משימות</option>
                            <option value="linked">משימות קשורות</option>
                        </select>
                    </div>
                </div>
                <div className="text-sm text-gray-600">
                    {sortedItems.length} פריטים • {view === 'today' ? 'היום' : view === 'week' ? 'השבוע' : 'החודש'}
                </div>
            </div>

            {/* Items list */}
            <div className="space-y-4">
                {sortedItems.length === 0 ? (
                    <div className="bg-white p-8 rounded-lg shadow text-center text-gray-500">
                        אין פריטים לתצוגה
                    </div>
                ) : (
                    sortedItems.map(item => {
                        const datetime = item.type === 'event' ? item.start_datetime : item.due_date;
                        const timeInfo = datetime ? formatDateTime(datetime) : null;

                        return (
                            <div key={`${item.type}-${item.id}`} className="bg-white rounded-lg shadow p-6 hover:shadow-md transition-shadow">
                                <div className="flex items-start justify-between">
                                    <div className="flex-1">
                                        <div className="flex items-center gap-3 mb-2">
                                            {item.type === 'event' ? (
                                                <span className="text-2xl">
                                                    {getEventTypeIcon(item.event_type)}
                                                </span>
                                            ) : (
                                                <CheckCircle className="w-6 h-6 text-gray-400" />
                                            )}
                                            <h3 className="text-lg font-medium text-gray-900">
                                                {item.title}
                                            </h3>
                                            <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs ${getStatusColor(item.status, item.type)}`}>
                                                {getStatusLabel(item.status, item.type)}
                                            </span>
                                            {item.type === 'task' && item.priority && (
                                                <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs ${getPriorityColor(item.priority)}`}>
                                                    {getPriorityLabel(item.priority)}
                                                </span>
                                            )}
                                        </div>

                                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm text-gray-600 mb-3">
                                            {timeInfo && (
                                                <div className="flex items-center gap-2">
                                                    <Clock className="w-4 h-4" />
                                                    <span>
                                                        {timeInfo.date} {timeInfo.time}
                                                    </span>
                                                </div>
                                            )}
                                            {item.customer_name && (
                                                <div className="flex items-center gap-2">
                                                    <span>👤</span>
                                                    <span>{item.customer_name}</span>
                                                </div>
                                            )}
                                            {item.type === 'event' && item.location && (
                                                <div className="flex items-center gap-2">
                                                    <span>📍</span>
                                                    <span>{item.location}</span>
                                                </div>
                                            )}
                                        </div>

                                        {/* Task progress bar */}
                                        {item.type === 'task' && (
                                            <div className="mb-3">
                                                <div className="flex items-center justify-between text-sm text-gray-600 mb-1">
                                                    <span>התקדמות</span>
                                                    <span>{item.progress}%</span>
                                                </div>
                                                <div className="w-full bg-gray-200 rounded-full h-2">
                                                    <div
                                                        className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                                        style={{ width: `${item.progress}%` }}
                                                    ></div>
                                                </div>
                                            </div>
                                        )}

                                        {/* Event-Task link info */}
                                        {item.type === 'task' && item.event_title && (
                                            <div className="flex items-center gap-2 text-sm text-blue-600 bg-blue-50 px-3 py-2 rounded">
                                                <Calendar className="w-4 h-4" />
                                                <span>קשור לאירוע: {item.event_title}</span>
                                                {item.event_start && (
                                                    <span className="text-gray-500">
                                                        ({formatDateTime(item.event_start).date} {formatDateTime(item.event_start).time})
                                                    </span>
                                                )}
                                            </div>
                                        )}
                                    </div>

                                    <div className="flex items-center gap-2">
                                        {item.type === 'event' ? (
                                            <>
                                                <a
                                                    href={`/events/${item.id}`}
                                                    className="p-2 text-gray-400 hover:text-blue-600 transition-colors"
                                                    title="צפה באירוע"
                                                >
                                                    <Eye className="w-4 h-4" />
                                                </a>
                                                <a
                                                    href={`/events/${item.id}/edit`}
                                                    className="p-2 text-gray-400 hover:text-yellow-600 transition-colors"
                                                    title="ערוך אירוע"
                                                >
                                                    <Edit className="w-4 h-4" />
                                                </a>
                                            </>
                                        ) : (
                                            <>
                                                <a
                                                    href={`/tasks/${item.id}`}
                                                    className="p-2 text-gray-400 hover:text-blue-600 transition-colors"
                                                    title="צפה במשימה"
                                                >
                                                    <Eye className="w-4 h-4" />
                                                </a>
                                                <a
                                                    href={`/tasks/${item.id}/edit`}
                                                    className="p-2 text-gray-400 hover:text-yellow-600 transition-colors"
                                                    title="ערוך משימה"
                                                >
                                                    <Edit className="w-4 h-4" />
                                                </a>
                                            </>
                                        )}
                                    </div>
                                </div>
                            </div>
                        );
                    })
                )}
            </div>

            {/* Quick actions */}
            <div className="bg-white p-6 rounded-lg shadow">
                <h3 className="text-lg font-semibold mb-4">פעולות מהירות</h3>
                <div className="flex flex-wrap gap-3">
                    <a
                        href="/events/new"
                        className="inline-flex items-center gap-2 px-4 py-2 bg-white border-2 border-blue-600 text-blue-600 rounded-md hover:bg-blue-50 hover:border-blue-700 transition-colors"
                    >
                        <Plus className="w-4 h-4" />
                        אירוע חדש
                    </a>
                    <a
                        href="/tasks/new"
                        className="inline-flex items-center gap-2 px-4 py-2 bg-white border-2 border-green-600 text-green-600 rounded-md hover:bg-green-50 hover:border-green-700 transition-colors"
                    >
                        <Plus className="w-4 h-4" />
                        משימה חדשה
                    </a>
                    <a
                        href="/events"
                        className="inline-flex items-center gap-2 px-4 py-2 bg-white border-2 border-gray-400 text-gray-700 rounded-md hover:bg-gray-50 hover:border-gray-500 transition-colors"
                    >
                        <Calendar className="w-4 h-4" />
                        כל האירועים
                    </a>
                    <a
                        href="/tasks"
                        className="inline-flex items-center gap-2 px-4 py-2 bg-white border-2 border-gray-400 text-gray-700 rounded-md hover:bg-gray-50 hover:border-gray-500 transition-colors"
                    >
                        <CheckCircle className="w-4 h-4" />
                        כל המשימות
                    </a>
                </div>
            </div>
        </div>
    );
}
