'use client';

import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Plus, CheckCircle, Edit, Calendar, AlertTriangle, Clock, PlayCircle, PauseCircle, Eye } from 'lucide-react';

interface Task {
    id: string;
    title: string;
    description?: string;
    status: string;
    priority: string;
    category: string;
    due_date: string;
    progress: number;
    domain_name?: string;
    project_name?: string;
}

interface CustomerTasksProps {
    customerId: string;
    customerName?: string;
}

export function CustomerTasks({ customerId, customerName }: CustomerTasksProps) {
    const [tasks, setTasks] = useState<Task[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    useEffect(() => {
        fetchTasks();
    }, [customerId]);

    const fetchTasks = async () => {
        try {
            setLoading(true);
            const response = await fetch(`/api/tasks?customer_id=${customerId}`);

            if (!response.ok) {
                throw new Error('שגיאה בטעינת משימות');
            }

            const data = await response.json();

            // Convert API data to our Task interface
            const tasksData: Task[] = data.map((task: any) => ({
                id: task.id.toString(),
                title: task.title,
                description: task.description,
                status: task.status,
                priority: task.priority,
                category: task.category,
                due_date: task.due_date || '',
                progress: task.progress || 0,
                domain_name: task.domain_name,
                project_name: task.project_name
            }));

            setTasks(tasksData);
        } catch (err) {
            console.error('Error fetching tasks:', err);
            setError(err instanceof Error ? err.message : 'שגיאה בטעינת משימות');
        } finally {
            setLoading(false);
        }
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString('he-IL');
    };

    const getDaysUntilDue = (dueDate: string) => {
        const today = new Date();
        const due = new Date(dueDate);
        const diffTime = due.getTime() - today.getTime();
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        return diffDays;
    };

    const getPriorityColor = (priority: string) => {
        switch (priority) {
            case 'urgent':
                return 'bg-red-100 text-red-800';
            case 'high':
                return 'bg-orange-100 text-orange-800';
            case 'medium':
                return 'bg-yellow-100 text-yellow-800';
            case 'low':
                return 'bg-green-100 text-green-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getPriorityText = (priority: string) => {
        switch (priority) {
            case 'urgent':
                return '🔴 דחוף';
            case 'high':
                return '🟠 גבוה';
            case 'medium':
                return '🟡 בינוני';
            case 'low':
                return '🟢 נמוך';
            default:
                return priority;
        }
    };

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'pending':
                return 'bg-gray-100 text-gray-800';
            case 'in_progress':
                return 'bg-blue-100 text-blue-800';
            case 'completed':
                return 'bg-green-100 text-green-800';
            case 'cancelled':
                return 'bg-red-100 text-red-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getStatusText = (status: string) => {
        switch (status) {
            case 'pending':
                return '⏰ ממתין';
            case 'in_progress':
                return '🔵 בתהליך';
            case 'completed':
                return '✅ הושלם';
            case 'cancelled':
                return '❌ בוטל';
            default:
                return status;
        }
    };

    const getCategoryText = (category: string) => {
        switch (category) {
            case 'domain':
                return '🌐 דומיין';
            case 'finance':
                return '💰 כספים';
            case 'maintenance':
                return '🔧 תחזוקה';
            case 'development':
                return '💻 פיתוח';
            case 'marketing':
                return '📢 שיווק';
            default:
                return category;
        }
    };

    const handleCompleteTask = async (taskId: string) => {
        try {
            const response = await fetch(`/api/tasks/${taskId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ status: 'completed' })
            });
            if (response.ok) {
                // Refresh tasks without page reload
                await fetchTasks();
            }
        } catch (error) {
            console.error('Error completing task:', error);
        }
    };

    const handleEditTask = (task: Task) => {
        window.location.href = `/tasks/${task.id}/edit`;
    };

    const handleCreateTask = () => {
        const params = new URLSearchParams({ customer_id: customerId });
        if (customerName) {
            params.append('customer_name', customerName);
        }
        window.location.href = `/tasks/new?${params.toString()}`;
    };

    const urgentTasks = tasks.filter(t => t.priority === 'urgent' && t.status !== 'completed');
    const activeTasks = tasks.filter(t => t.status === 'in_progress');
    const pendingTasks = tasks.filter(t => t.status === 'pending');

    if (loading) {
        return (
            <div className="space-y-4">
                <div className="flex justify-between items-center">
                    <h2 className="text-xl font-bold">📋 משימות</h2>
                </div>
                <div className="text-center py-8">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
                    <p className="text-gray-600">טוען משימות...</p>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="space-y-4">
                <div className="flex justify-between items-center">
                    <h2 className="text-xl font-bold">📋 משימות</h2>
                </div>
                <Card>
                    <CardContent className="py-12 text-center text-red-500">
                        <AlertTriangle className="h-12 w-12 mx-auto mb-4" />
                        <p className="text-xl mb-2">שגיאה בטעינת משימות</p>
                        <p className="text-sm">{error}</p>
                        <Button onClick={fetchTasks} className="mt-4">
                            🔄 נסה שוב
                        </Button>
                    </CardContent>
                </Card>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <h2 className="text-xl font-bold">📋 משימות</h2>
                <Button onClick={handleCreateTask}>
                    <Plus className="w-4 h-4 ml-2" />
                    משימה חדשה
                </Button>
            </div>

            {tasks.length === 0 ? (
                <Card>
                    <CardContent className="py-12 text-center text-gray-500">
                        <p className="text-xl mb-4">אין משימות עדיין</p>
                        <p className="text-sm mb-4">צור משימה ראשונה כדי להתחיל</p>
                        <Button onClick={handleCreateTask}>
                            <Plus className="w-4 h-4 ml-2" />
                            צור משימה ראשונה
                        </Button>
                    </CardContent>
                </Card>
            ) : (
                <div className="space-y-6">
                    {/* דחופות */}
                    {urgentTasks.length > 0 && (
                        <div>
                            <h3 className="text-lg font-semibold mb-3 text-red-600">
                                🔴 דחופות ({urgentTasks.length})
                            </h3>
                            <div className="space-y-3">
                                {urgentTasks.map(task => (
                                    <TaskCard key={task.id} task={task} onComplete={handleCompleteTask} onEdit={handleEditTask} />
                                ))}
                            </div>
                        </div>
                    )}

                    {/* בתהליך */}
                    {activeTasks.length > 0 && (
                        <div>
                            <h3 className="text-lg font-semibold mb-3">
                                🔵 בתהליך ({activeTasks.length})
                            </h3>
                            <div className="space-y-3">
                                {activeTasks.map(task => (
                                    <TaskCard key={task.id} task={task} onComplete={handleCompleteTask} onEdit={handleEditTask} />
                                ))}
                            </div>
                        </div>
                    )}

                    {/* ממתינות */}
                    {pendingTasks.length > 0 && (
                        <div>
                            <h3 className="text-lg font-semibold mb-3">
                                ⏰ ממתינות ({pendingTasks.length})
                            </h3>
                            <div className="space-y-3">
                                {pendingTasks.map(task => (
                                    <TaskCard key={task.id} task={task} onComplete={handleCompleteTask} onEdit={handleEditTask} />
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            )}
        </div>
    );
}

function TaskCard({
    task,
    onComplete,
    onEdit
}: {
    task: Task;
    onComplete: (taskId: string) => void;
    onEdit: (task: Task) => void;
}) {
    const getDaysUntilDue = (dueDate: string) => {
        const today = new Date();
        const due = new Date(dueDate);
        const diffTime = due.getTime() - today.getTime();
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        return diffDays;
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString('he-IL');
    };

    const getPriorityColor = (priority: string) => {
        switch (priority) {
            case 'urgent':
                return 'bg-red-100 text-red-800';
            case 'high':
                return 'bg-orange-100 text-orange-800';
            case 'medium':
                return 'bg-yellow-100 text-yellow-800';
            case 'low':
                return 'bg-green-100 text-green-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'pending':
                return 'bg-gray-100 text-gray-800';
            case 'in_progress':
                return 'bg-blue-100 text-blue-800';
            case 'completed':
                return 'bg-green-100 text-green-800';
            case 'cancelled':
                return 'bg-red-100 text-red-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getStatusText = (status: string) => {
        switch (status) {
            case 'pending':
                return '⏰ ממתין';
            case 'in_progress':
                return '🔵 בתהליך';
            case 'completed':
                return '✅ הושלם';
            case 'cancelled':
                return '❌ בוטל';
            default:
                return status;
        }
    };

    const getCategoryText = (category: string) => {
        switch (category) {
            case 'domain':
                return '🌐 דומיין';
            case 'finance':
                return '💰 כספים';
            case 'maintenance':
                return '🔧 תחזוקה';
            case 'development':
                return '💻 פיתוח';
            case 'marketing':
                return '📢 שיווק';
            default:
                return category;
        }
    };

    const daysUntilDue = getDaysUntilDue(task.due_date);
    const isOverdue = daysUntilDue < 0;
    const isDueSoon = daysUntilDue <= 3 && daysUntilDue >= 0;

    return (
        <Card className={isOverdue ? 'border-red-200 bg-red-50' : isDueSoon ? 'border-orange-200 bg-orange-50' : ''}>
            <CardHeader>
                <div className="flex items-start justify-between">
                    {/* כפתורים בצד שמאל */}
                    <div className="flex gap-2">
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={() => window.location.href = `/tasks/${task.id}`}
                        >
                            <Eye className="w-4 h-4 ml-1" />
                            צפייה
                        </Button>
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={() => onEdit(task)}
                        >
                            <Edit className="w-4 h-4 ml-1" />
                            ערוך
                        </Button>
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={() => onComplete(task.id)}
                            disabled={task.status === 'completed'}
                        >
                            <CheckCircle className="w-4 h-4 ml-1" />
                            הושלם
                        </Button>
                    </div>
                    
                    {/* טקסט בצד ימין */}
                    <div className="flex-1 text-right">
                        <div className="flex items-center gap-2 mb-2 justify-end">
                            <Badge className={getStatusColor(task.status)}>
                                {getStatusText(task.status)}
                            </Badge>
                            <Badge className={getPriorityColor(task.priority)}>
                                {task.priority === 'urgent' ? '🔴 דחוף' :
                                    task.priority === 'high' ? '🟠 גבוה' :
                                        task.priority === 'medium' ? '🟡 בינוני' : '🟢 נמוך'}
                            </Badge>
                            <CardTitle className="text-base">#{task.id} - {task.title}</CardTitle>
                        </div>
                        <div className="flex items-center gap-4 text-sm text-gray-600 justify-end">
                            {isOverdue && (
                                <span className="text-red-600 font-bold">(איחור!)</span>
                            )}
                            {isDueSoon && !isOverdue && (
                                <span className="text-orange-600 font-bold">(קרוב למועד!)</span>
                            )}
                            <span>{getCategoryText(task.category)}</span>
                            <span>📅 {formatDate(task.due_date)}</span>
                            {task.domain_name && (
                                <span>🌐 {task.domain_name}</span>
                            )}
                        </div>
                    </div>
                </div>
            </CardHeader>
            {task.progress > 0 && (
                <CardContent>
                    <div className="flex items-center gap-3">
                        <Progress value={task.progress} className="flex-1" />
                        <span className="text-sm text-gray-600">{task.progress}%</span>
                    </div>
                </CardContent>
            )}
        </Card>
    );
}
