'use client';

import React, { useState } from 'react';
import { Customer } from '@/src/types/customer';
import { CheckSquare, Square, Edit, Trash2, MoreVertical, FolderOpen, Eye } from 'lucide-react';

interface CustomerTableProps {
    customers: Customer[];
}

export const CustomerTable: React.FC<CustomerTableProps> = ({
    customers
}) => {
    const [selectedCustomers, setSelectedCustomers] = useState<string[]>([]);
    const [pageSize, setPageSize] = useState(10);
    const [currentPage, setCurrentPage] = useState(1);

    const handleSelectAll = () => {
        if (selectedCustomers.length === customers.length) {
            setSelectedCustomers([]);
        } else {
            setSelectedCustomers(customers.map(c => c.id));
        }
    };

    const handleSelectCustomer = (customerId: string) => {
        if (selectedCustomers.includes(customerId)) {
            setSelectedCustomers(selectedCustomers.filter(id => id !== customerId));
        } else {
            setSelectedCustomers([...selectedCustomers, customerId]);
        }
    };

    const handleBulkDelete = async () => {
        if (selectedCustomers.length === 0) return;

        if (confirm(`האם אתה בטוח שברצונך למחוק ${selectedCustomers.length} לקוחות?`)) {
            try {
                for (const customerId of selectedCustomers) {
                    const response = await fetch(`/api/customers/${customerId}`, {
                        method: 'DELETE',
                    });
                    if (!response.ok) {
                        throw new Error('Failed to delete customer');
                    }
                }
                setSelectedCustomers([]);
                window.location.reload();
            } catch (error) {
                console.error('Error deleting customers:', error);
                alert('שגיאה במחיקת הלקוחות');
            }
        }
    };

    // Pagination logic
    const startIndex = (currentPage - 1) * pageSize;
    const endIndex = startIndex + pageSize;
    const paginatedCustomers = customers.slice(startIndex, endIndex);
    const totalPages = Math.ceil(customers.length / pageSize);
    const getStatusText = (status: string) => {
        switch (status) {
            case 'active':
                return 'פעיל';
            case 'inactive':
                return 'לא פעיל';
            case 'lead':
                return 'ליד';
            default:
                return status;
        }
    };

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'active':
                return 'bg-green-100 text-green-800';
            case 'inactive':
                return 'bg-gray-100 text-gray-800';
            case 'lead':
                return 'bg-blue-100 text-blue-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getStageText = (stage: string) => {
        switch (stage) {
            case 'quote':
                return 'הצעת מחיר';
            case 'project':
                return 'פרויקט';
            case 'follow_up':
                return 'מעקב';
            case 'closed':
                return 'סגור';
            default:
                return stage;
        }
    };

    const getStageColor = (stage: string) => {
        switch (stage) {
            case 'quote':
                return 'bg-yellow-100 text-yellow-800';
            case 'project':
                return 'bg-blue-100 text-blue-800';
            case 'follow_up':
                return 'bg-orange-100 text-orange-800';
            case 'closed':
                return 'bg-green-100 text-green-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    return (
        <div className="bg-white rounded-lg shadow overflow-hidden">
            {/* Controls */}
            <div className="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
                <div className="flex items-center space-x-4 space-x-reverse">
                    <span className="text-sm text-gray-700">
                        הצג {pageSize} רשומות לעמוד
                    </span>
                    <select
                        value={pageSize}
                        onChange={(e) => {
                            setPageSize(Number(e.target.value));
                            setCurrentPage(1);
                        }}
                        className="border border-gray-300 rounded-md px-3 py-1 text-sm"
                        title="בחר מספר רשומות לעמוד"
                        aria-label="בחר מספר רשומות לעמוד"
                    >
                        <option value={10}>10</option>
                        <option value={50}>50</option>
                        <option value={100}>100</option>
                    </select>
                </div>

                {selectedCustomers.length > 0 && (
                    <div className="flex items-center space-x-2 space-x-reverse">
                        <span className="text-sm text-gray-700">
                            {selectedCustomers.length} נבחרו
                        </span>
                        <button
                            onClick={handleBulkDelete}
                            className="px-3 py-1 bg-red-600 text-white rounded-md text-sm hover:bg-red-700 flex items-center space-x-1 space-x-reverse"
                        >
                            <Trash2 className="w-4 h-4" />
                            <span>מחק נבחרים</span>
                        </button>
                    </div>
                )}
            </div>

            <div className="overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                        <tr>
                            <th className="px-6 py-3 text-center">
                                <button
                                    onClick={handleSelectAll}
                                    className="text-gray-400 hover:text-gray-600"
                                >
                                    {selectedCustomers.length === customers.length ? (
                                        <CheckSquare className="w-5 h-5" />
                                    ) : (
                                        <Square className="w-5 h-5" />
                                    )}
                                </button>
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                שם
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                אימייל
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                טלפון
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                חברה
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                סטטוס
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                שלב
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                ערך
                            </th>
                            <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                פעולות
                            </th>
                        </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                        {paginatedCustomers.map((customer) => (
                            <tr key={customer.id} className={`hover:bg-gray-50 ${selectedCustomers.includes(customer.id) ? 'bg-blue-50' : ''}`}>
                                <td className="px-6 py-4 whitespace-nowrap text-center">
                                    <button
                                        onClick={() => handleSelectCustomer(customer.id)}
                                        className="text-gray-400 hover:text-gray-600"
                                    >
                                        {selectedCustomers.includes(customer.id) ? (
                                            <CheckSquare className="w-5 h-5 text-blue-600" />
                                        ) : (
                                            <Square className="w-5 h-5" />
                                        )}
                                    </button>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <a
                                        href={`/customers/${customer.id}`}
                                        className="text-sm font-medium text-blue-600 hover:text-blue-800 hover:underline"
                                    >
                                        {customer.name}
                                    </a>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">{customer.email}</div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">{customer.phone || '-'}</div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">{customer.company || '-'}</div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(customer.status)}`}>
                                        {getStatusText(customer.status)}
                                    </span>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStageColor(customer.stage)}`}>
                                        {getStageText(customer.stage)}
                                    </span>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">
                                        ₪{customer.value ? customer.value.toLocaleString() : '0'}
                                    </div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div className="flex space-x-2 space-x-reverse">
                                        <a
                                            href={`/customers/${customer.id}`}
                                            className="p-1 text-purple-600 hover:text-purple-900 hover:bg-purple-100 rounded"
                                            title="צפה בפרטים"
                                        >
                                            <Eye className="w-4 h-4" />
                                        </a>
                                        <a
                                            href={`/projects?customer_id=${customer.id}`}
                                            className="p-1 text-green-600 hover:text-green-900 hover:bg-green-100 rounded"
                                            title="פרויקטים"
                                        >
                                            <FolderOpen className="w-4 h-4" />
                                        </a>
                                        <a
                                            href={`/customers/${customer.id}/edit`}
                                            className="p-1 text-blue-600 hover:text-blue-900 hover:bg-blue-100 rounded"
                                            title="עריכה"
                                        >
                                            <Edit className="w-4 h-4" />
                                        </a>
                                        <button
                                            onClick={async () => {
                                                if (confirm('האם אתה בטוח שברצונך למחוק את הלקוח?')) {
                                                    try {
                                                        const response = await fetch(`/api/customers/${customer.id}`, {
                                                            method: 'DELETE',
                                                        });

                                                        if (!response.ok) {
                                                            throw new Error('Failed to delete customer');
                                                        }

                                                        // Refresh the page to show updated data
                                                        window.location.reload();
                                                    } catch (error) {
                                                        console.error('Error deleting customer:', error);
                                                        alert('שגיאה במחיקת הלקוח');
                                                    }
                                                }
                                            }}
                                            className="p-1 text-red-600 hover:text-red-900 hover:bg-red-100 rounded"
                                            title="מחיקה"
                                        >
                                            <Trash2 className="w-4 h-4" />
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>

            {/* Pagination */}
            {totalPages > 1 && (
                <div className="px-6 py-4 border-t border-gray-200 flex justify-between items-center">
                    <div className="text-sm text-gray-700">
                        מציג {startIndex + 1} עד {Math.min(endIndex, customers.length)} מתוך {customers.length} רשומות
                    </div>
                    <div className="flex space-x-2 space-x-reverse">
                        <button
                            onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                            disabled={currentPage === 1}
                            className="px-3 py-1 text-sm border border-gray-300 rounded-md disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
                        >
                            קודם
                        </button>

                        {Array.from({ length: totalPages }, (_, i) => i + 1).map(page => (
                            <button
                                key={page}
                                onClick={() => setCurrentPage(page)}
                                className={`px-3 py-1 text-sm border rounded-md ${currentPage === page
                                    ? 'bg-blue-600 text-white border-blue-600'
                                    : 'border-gray-300 hover:bg-gray-50'
                                    }`}
                            >
                                {page}
                            </button>
                        ))}

                        <button
                            onClick={() => setCurrentPage(prev => Math.min(totalPages, prev + 1))}
                            disabled={currentPage === totalPages}
                            className="px-3 py-1 text-sm border border-gray-300 rounded-md disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
                        >
                            הבא
                        </button>
                    </div>
                </div>
            )}
        </div>
    );
};
