'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Plus } from 'lucide-react';

interface Customer {
    id: string;
    name: string;
    email?: string;
}

interface CustomerSelectProps {
    value?: string;
    onChange: (value: string) => void;
    placeholder?: string;
}

export function CustomerSelect({
    value,
    onChange,
    placeholder = "בחר לקוח"
}: CustomerSelectProps) {
    const [customers, setCustomers] = useState<Customer[]>([]);
    const [loading, setLoading] = useState(true);
    const router = useRouter();

    useEffect(() => {
        const fetchCustomers = async () => {
            try {
                const response = await fetch('/api/customers');
                if (response.ok) {
                    const data = await response.json();
                    setCustomers(data);
                }
            } catch (error) {
                console.error('Error fetching customers:', error);
            } finally {
                setLoading(false);
            }
        };

        fetchCustomers();
    }, []);

    if (loading) {
        return (
            <Select disabled>
                <SelectTrigger>
                    <SelectValue placeholder="טוען לקוחות..." />
                </SelectTrigger>
            </Select>
        );
    }

    const handleAddCustomer = () => {
        router.push('/customers/add');
    };

    return (
        <div className="space-y-2">
            <Select value={value} onValueChange={onChange}>
                <SelectTrigger>
                    <SelectValue placeholder={placeholder} />
                </SelectTrigger>
                <SelectContent>
                    {customers.map((customer) => (
                        <SelectItem key={customer.id} value={customer.id}>
                            {customer.name}
                            {customer.email && ` (${customer.email})`}
                        </SelectItem>
                    ))}
                </SelectContent>
            </Select>

            <Button
                type="button"
                variant="outline"
                size="sm"
                onClick={handleAddCustomer}
                className="w-full"
            >
                <Plus className="w-4 h-4 ml-2" />
                הוסף לקוח חדש
            </Button>
        </div>
    );
}
