'use client';

import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Plus, Eye, Edit, RefreshCw, AlertTriangle, CheckCircle } from 'lucide-react';
import Link from 'next/link';

interface Domain {
    id: string;
    domain_name: string;
    status: string;
    expiry_date: string;
    ssl_status: string;
    active_tasks_count: number;
}

interface CustomerDomainsProps {
    customerId: string;
}

export function CustomerDomains({ customerId }: CustomerDomainsProps) {
    const [domains, setDomains] = useState<Domain[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    useEffect(() => {
        fetchDomains();
    }, [customerId]);

    const fetchDomains = async () => {
        try {
            setLoading(true);
            const response = await fetch(`/api/domains?customer_id=${customerId}`);

            if (!response.ok) {
                throw new Error('שגיאה בטעינת דומיינים');
            }

            const data = await response.json();
            setDomains(data.domains || []);
        } catch (err) {
            console.error('Error fetching domains:', err);
            setError(err instanceof Error ? err.message : 'שגיאה בטעינת דומיינים');
        } finally {
            setLoading(false);
        }
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString('he-IL');
    };

    const getDaysUntilExpiry = (expiryDate: string) => {
        const today = new Date();
        const expiry = new Date(expiryDate);
        const diffTime = expiry.getTime() - today.getTime();
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        return diffDays;
    };

    const getExpiryClass = (expiryDate: string) => {
        const days = getDaysUntilExpiry(expiryDate);
        if (days < 0) return 'text-red-600 font-bold';
        if (days < 30) return 'text-orange-600 font-bold';
        if (days < 90) return 'text-yellow-600';
        return 'text-gray-600';
    };

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'active':
                return 'bg-green-100 text-green-800';
            case 'inactive':
                return 'bg-gray-100 text-gray-800';
            case 'expired':
                return 'bg-red-100 text-red-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getStatusText = (status: string) => {
        switch (status) {
            case 'active':
                return '🟢 פעיל';
            case 'inactive':
                return '⚫ לא פעיל';
            case 'expired':
                return '🔴 פג';
            default:
                return status;
        }
    };

    const handleCreateTask = (domainId: string) => {
        // Navigate to create task page with domain pre-selected
        window.location.href = `/tasks/new?domain_id=${domainId}`;
    };

    const handleRenewDomain = (domainId: string) => {
        // Open renew domain modal
        alert(`חידוש דומיין: ${domainId}`);
        // TODO: Implement proper domain renewal modal
    };

    const handleAddDomain = () => {
        // Open add domain modal
        alert('הוספת דומיין חדש');
        // TODO: Implement proper add domain modal
    };

    const handleEditDomain = (domainId: string) => {
        // Open edit domain modal
        alert(`עריכת דומיין: ${domainId}`);
        // TODO: Implement proper edit domain modal
    };

    if (loading) {
        return (
            <div className="space-y-4">
                <div className="flex justify-between items-center">
                    <h2 className="text-xl font-bold">🌐 דומיינים</h2>
                </div>
                <div className="text-center py-8">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
                    <p className="text-gray-600">טוען דומיינים...</p>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="space-y-4">
                <div className="flex justify-between items-center">
                    <h2 className="text-xl font-bold">🌐 דומיינים</h2>
                </div>
                <Card>
                    <CardContent className="py-12 text-center text-red-500">
                        <AlertTriangle className="h-12 w-12 mx-auto mb-4" />
                        <p className="text-xl mb-2">שגיאה בטעינת דומיינים</p>
                        <p className="text-sm">{error}</p>
                        <Button onClick={fetchDomains} className="mt-4">
                            🔄 נסה שוב
                        </Button>
                    </CardContent>
                </Card>
            </div>
        );
    }

    return (
        <div className="space-y-4">
            <div className="flex justify-between items-center">
                <h2 className="text-xl font-bold">🌐 דומיינים</h2>
                <Button onClick={handleAddDomain}>
                    <Plus className="w-4 h-4 ml-2" />
                    דומיין חדש
                </Button>
            </div>

            {domains.length === 0 ? (
                <Card>
                    <CardContent className="py-12 text-center text-gray-500">
                        <p className="text-xl mb-4">אין דומיינים עדיין</p>
                        <p className="text-sm mb-4">הוסף דומיין ראשון כדי להתחיל</p>
                        <Button onClick={handleAddDomain}>
                            <Plus className="w-4 h-4 ml-2" />
                            הוסף דומיין ראשון
                        </Button>
                    </CardContent>
                </Card>
            ) : (
                <div className="space-y-4">
                    {domains.map((domain) => {
                        const daysUntilExpiry = getDaysUntilExpiry(domain.expiry_date);
                        const isExpiringSoon = daysUntilExpiry < 30 && daysUntilExpiry >= 0;
                        const isExpired = daysUntilExpiry < 0;

                        return (
                            <Card key={domain.id} className={isExpiringSoon || isExpired ? 'border-orange-200 bg-orange-50' : ''}>
                                <CardHeader>
                                    <div className="flex items-center justify-between">
                                        <div className="flex items-center gap-3">
                                            <CardTitle className="text-lg">
                                                🌐 {domain.domain_name}
                                            </CardTitle>
                                            <Badge className={getStatusColor(domain.status)}>
                                                {getStatusText(domain.status)}
                                            </Badge>
                                            {domain.ssl_status === 'active' && (
                                                <Badge variant="outline" className="bg-green-50 text-green-700 border-green-200">
                                                    <CheckCircle className="w-3 h-3 ml-1" />
                                                    SSL
                                                </Badge>
                                            )}
                                        </div>
                                        <div className="flex gap-2">
                                            <Link href={`/domains/${domain.id}`}>
                                                <Button variant="outline" size="sm">
                                                    <Eye className="w-4 h-4 ml-1" />
                                                    צפה
                                                </Button>
                                            </Link>
                                            <Button variant="outline" size="sm" onClick={() => handleEditDomain(domain.id)}>
                                                <Edit className="w-4 h-4 ml-1" />
                                                ערוך
                                            </Button>
                                        </div>
                                    </div>
                                </CardHeader>
                                <CardContent>
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <p className="text-sm text-gray-600">פקיעה:</p>
                                            <p className={getExpiryClass(domain.expiry_date)}>
                                                📅 {formatDate(domain.expiry_date)}
                                                {isExpiringSoon && (
                                                    <span className="text-orange-600 font-bold">
                                                        {' '}(עוד {daysUntilExpiry} יום)
                                                    </span>
                                                )}
                                                {isExpired && (
                                                    <span className="text-red-600 font-bold">
                                                        {' '}(פג!)
                                                    </span>
                                                )}
                                            </p>
                                        </div>
                                        <div>
                                            <p className="text-sm text-gray-600">משימות פעילות:</p>
                                            <p className="font-medium">
                                                📋 {domain.active_tasks_count} משימות
                                            </p>
                                        </div>
                                    </div>

                                    {domain.active_tasks_count > 0 && (
                                        <div className="mt-3 pt-3 border-t">
                                            <Button
                                                variant="link"
                                                onClick={() => handleCreateTask(domain.id)}
                                                className="p-0 text-blue-600 hover:text-blue-800"
                                            >
                                                📋 הצג משימות ({domain.active_tasks_count})
                                            </Button>
                                        </div>
                                    )}

                                    <div className="mt-3 flex gap-2">
                                        <Button
                                            variant="outline"
                                            size="sm"
                                            onClick={() => handleCreateTask(domain.id)}
                                        >
                                            📋 צור משימה
                                        </Button>
                                        <Button
                                            variant="outline"
                                            size="sm"
                                            onClick={() => handleRenewDomain(domain.id)}
                                        >
                                            <RefreshCw className="w-4 h-4 ml-1" />
                                            חדש דומיין
                                        </Button>
                                    </div>
                                </CardContent>
                            </Card>
                        );
                    })}
                </div>
            )}
        </div>
    );
}
