'use client';

import React, { useEffect, useState } from 'react';
import { ChevronLeft, ChevronRight, Calendar, CheckCircle, Clock, MapPin, Users } from 'lucide-react';

interface CalendarEvent {
    id: number;
    title: string;
    start_datetime: string;
    end_datetime?: string;
    event_type: string;
    status: string;
    customer_name?: string;
    location?: string;
    type: 'event';
}

interface CalendarTask {
    id: number;
    title: string;
    due_date?: string;
    status: string;
    priority: string;
    progress: number;
    customer_name?: string;
    event_title?: string;
    type: 'task';
}

interface CalendarItem extends CalendarEvent, CalendarTask {
    type: 'event' | 'task';
}

export default function CalendarView() {
    const [currentDate, setCurrentDate] = useState(new Date());
    const [items, setItems] = useState<CalendarItem[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [view, setView] = useState<'month' | 'week' | 'day'>('month');

    // Load calendar data
    const loadCalendarData = async (date: Date) => {
        setLoading(true);
        try {
            const startDate = new Date(date);
            const endDate = new Date(date);

            if (view === 'month') {
                startDate.setDate(1);
                endDate.setMonth(endDate.getMonth() + 1);
                endDate.setDate(0);
            } else if (view === 'week') {
                const day = startDate.getDay();
                startDate.setDate(startDate.getDate() - day);
                endDate.setDate(startDate.getDate() + 6);
            } else {
                // day view
                endDate.setDate(startDate.getDate() + 1);
            }

            const params = new URLSearchParams({
                start_date: startDate.toISOString().split('T')[0],
                end_date: endDate.toISOString().split('T')[0]
            });

            const response = await fetch(`/api/calendar?${params}`);
            if (response.ok) {
                const data = await response.json();
                const calendarItems: CalendarItem[] = [
                    ...(data.events || []).map((event: any) => ({
                        ...event,
                        type: 'event' as const
                    })),
                    ...(data.tasks || []).map((task: any) => ({
                        ...task,
                        type: 'task' as const
                    }))
                ];
                setItems(calendarItems);
            } else {
                setError('שגיאה בטעינת נתוני הלוח שנה');
            }
        } catch (err) {
            setError('שגיאה בטעינת נתוני הלוח שנה');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        loadCalendarData(currentDate);
    }, [currentDate, view]);

    // Navigation
    const goToPrevious = () => {
        const newDate = new Date(currentDate);
        if (view === 'month') {
            newDate.setMonth(newDate.getMonth() - 1);
        } else if (view === 'week') {
            newDate.setDate(newDate.getDate() - 7);
        } else {
            newDate.setDate(newDate.getDate() - 1);
        }
        setCurrentDate(newDate);
    };

    const goToNext = () => {
        const newDate = new Date(currentDate);
        if (view === 'month') {
            newDate.setMonth(newDate.getMonth() + 1);
        } else if (view === 'week') {
            newDate.setDate(newDate.getDate() + 7);
        } else {
            newDate.setDate(newDate.getDate() + 1);
        }
        setCurrentDate(newDate);
    };

    const goToToday = () => {
        setCurrentDate(new Date());
    };

    // Get items for a specific date
    const getItemsForDate = (date: Date) => {
        const dateStr = date.toISOString().split('T')[0];
        return items.filter(item => {
            const itemDate = item.type === 'event' ? item.start_datetime : item.due_date;
            return itemDate && itemDate.startsWith(dateStr);
        });
    };

    // Get event type icon
    const getEventTypeIcon = (type: string) => {
        const icons: Record<string, string> = {
            meeting: '🤝',
            call: '📞',
            presentation: '📊',
            webinar: '💻',
            deadline: '⏰',
            reminder: '🔔',
            other: '📌'
        };
        return icons[type] || '📌';
    };

    // Get status color
    const getStatusColor = (status: string, type: 'event' | 'task') => {
        if (type === 'event') {
            const colors: Record<string, string> = {
                scheduled: 'bg-blue-100 text-blue-800 border-blue-200',
                in_progress: 'bg-yellow-100 text-yellow-800 border-yellow-200',
                completed: 'bg-green-100 text-green-800 border-green-200',
                cancelled: 'bg-red-100 text-red-800 border-red-200',
                postponed: 'bg-gray-100 text-gray-800 border-gray-200'
            };
            return colors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
        } else {
            const colors: Record<string, string> = {
                pending: 'bg-yellow-100 text-yellow-800 border-yellow-200',
                in_progress: 'bg-blue-100 text-blue-800 border-blue-200',
                completed: 'bg-green-100 text-green-800 border-green-200',
                cancelled: 'bg-red-100 text-red-800 border-red-200',
                on_hold: 'bg-purple-100 text-purple-800 border-purple-200'
            };
            return colors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    // Get priority color for tasks
    const getPriorityColor = (priority: string) => {
        const colors: Record<string, string> = {
            urgent: 'border-l-red-500',
            high: 'border-l-orange-500',
            medium: 'border-l-yellow-500',
            low: 'border-l-green-500'
        };
        return colors[priority] || 'border-l-gray-500';
    };

    // Format time
    const formatTime = (datetime: string) => {
        const date = new Date(datetime);
        return date.toLocaleTimeString('he-IL', { hour: '2-digit', minute: '2-digit' });
    };

    // Generate calendar days
    const generateCalendarDays = () => {
        const year = currentDate.getFullYear();
        const month = currentDate.getMonth();
        const firstDay = new Date(year, month, 1);
        const lastDay = new Date(year, month + 1, 0);
        const startDate = new Date(firstDay);
        startDate.setDate(startDate.getDate() - firstDay.getDay());

        const days = [];
        for (let i = 0; i < 42; i++) {
            const date = new Date(startDate);
            date.setDate(startDate.getDate() + i);
            days.push(date);
        }
        return days;
    };

    // Render month view
    const renderMonthView = () => {
        const days = generateCalendarDays();
        const monthNames = [
            'ינואר', 'פברואר', 'מרץ', 'אפריל', 'מאי', 'יוני',
            'יולי', 'אוגוסט', 'ספטמבר', 'אוקטובר', 'נובמבר', 'דצמבר'
        ];
        const dayNames = ['א', 'ב', 'ג', 'ד', 'ה', 'ו', 'ש'];

        return (
            <div className="bg-white rounded-lg shadow overflow-hidden">
                {/* Header */}
                <div className="flex items-center justify-between p-4 border-b">
                    <h2 className="text-xl font-semibold">
                        {monthNames[currentDate.getMonth()]} {currentDate.getFullYear()}
                    </h2>
                    <div className="flex items-center gap-2">
                        <button
                            onClick={goToPrevious}
                            className="p-2 hover:bg-gray-100 rounded-md transition-colors"
                            title="קודם"
                            aria-label="קודם"
                        >
                            <ChevronRight className="w-4 h-4" />
                        </button>
                        <button
                            onClick={goToToday}
                            className="px-3 py-1 text-sm bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors"
                        >
                            היום
                        </button>
                        <button
                            onClick={goToNext}
                            className="p-2 hover:bg-gray-100 rounded-md transition-colors"
                            title="הבא"
                            aria-label="הבא"
                        >
                            <ChevronLeft className="w-4 h-4" />
                        </button>
                    </div>
                </div>

                {/* Day names */}
                <div className="grid grid-cols-7 border-b">
                    {dayNames.map(day => (
                        <div key={day} className="p-3 text-center text-sm font-medium text-gray-500 bg-gray-50">
                            {day}
                        </div>
                    ))}
                </div>

                {/* Calendar grid */}
                <div className="grid grid-cols-7">
                    {days.map((date, index) => {
                        const isCurrentMonth = date.getMonth() === currentDate.getMonth();
                        const isToday = date.toDateString() === new Date().toDateString();
                        const dayItems = getItemsForDate(date);

                        return (
                            <div
                                key={index}
                                className={`min-h-24 p-2 border-r border-b ${isCurrentMonth ? 'bg-white' : 'bg-gray-50'
                                    } ${isToday ? 'bg-blue-50' : ''}`}
                            >
                                <div className={`text-sm font-medium mb-1 ${isCurrentMonth ? 'text-gray-900' : 'text-gray-400'
                                    } ${isToday ? 'text-blue-600' : ''}`}>
                                    {date.getDate()}
                                </div>
                                <div className="space-y-1">
                                    {dayItems.slice(0, 3).map(item => (
                                        <div
                                            key={`${item.type}-${item.id}`}
                                            className={`text-xs p-1 rounded border-l-2 ${getStatusColor(item.status, item.type)} ${item.type === 'task' ? getPriorityColor(item.priority) : ''
                                                }`}
                                        >
                                            <div className="flex items-center gap-1">
                                                {item.type === 'event' ? (
                                                    <span>{getEventTypeIcon(item.event_type)}</span>
                                                ) : (
                                                    <CheckCircle className="w-3 h-3" />
                                                )}
                                                <span className="truncate">{item.title}</span>
                                            </div>
                                            {item.type === 'event' && item.start_datetime && (
                                                <div className="text-xs text-gray-500">
                                                    {formatTime(item.start_datetime)}
                                                </div>
                                            )}
                                        </div>
                                    ))}
                                    {dayItems.length > 3 && (
                                        <div className="text-xs text-gray-500">
                                            +{dayItems.length - 3} נוספים
                                        </div>
                                    )}
                                </div>
                            </div>
                        );
                    })}
                </div>
            </div>
        );
    };

    // Render week view
    const renderWeekView = () => {
        const startOfWeek = new Date(currentDate);
        const day = startOfWeek.getDay();
        startOfWeek.setDate(startOfWeek.getDate() - day);

        const weekDays = [];
        for (let i = 0; i < 7; i++) {
            const date = new Date(startOfWeek);
            date.setDate(startOfWeek.getDate() + i);
            weekDays.push(date);
        }

        return (
            <div className="bg-white rounded-lg shadow overflow-hidden">
                {/* Header */}
                <div className="flex items-center justify-between p-4 border-b">
                    <h2 className="text-xl font-semibold">
                        שבוע {startOfWeek.getDate()}-{weekDays[6].getDate()} {startOfWeek.toLocaleDateString('he-IL', { month: 'long' })}
                    </h2>
                    <div className="flex items-center gap-2">
                        <button
                            onClick={goToPrevious}
                            className="p-2 hover:bg-gray-100 rounded-md transition-colors"
                            title="קודם"
                            aria-label="קודם"
                        >
                            <ChevronRight className="w-4 h-4" />
                        </button>
                        <button
                            onClick={goToToday}
                            className="px-3 py-1 text-sm bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors"
                        >
                            היום
                        </button>
                        <button
                            onClick={goToNext}
                            className="p-2 hover:bg-gray-100 rounded-md transition-colors"
                            title="הבא"
                            aria-label="הבא"
                        >
                            <ChevronLeft className="w-4 h-4" />
                        </button>
                    </div>
                </div>

                {/* Week grid */}
                <div className="grid grid-cols-7">
                    {weekDays.map((date, index) => {
                        const isToday = date.toDateString() === new Date().toDateString();
                        const dayItems = getItemsForDate(date);

                        return (
                            <div key={index} className={`p-4 border-r ${isToday ? 'bg-blue-50' : 'bg-white'}`}>
                                <div className={`text-sm font-medium mb-3 ${isToday ? 'text-blue-600' : 'text-gray-900'
                                    }`}>
                                    {date.toLocaleDateString('he-IL', { weekday: 'long' })}
                                    <br />
                                    {date.getDate()}
                                </div>
                                <div className="space-y-2">
                                    {dayItems.map(item => (
                                        <div
                                            key={`${item.type}-${item.id}`}
                                            className={`text-xs p-2 rounded border-l-2 ${getStatusColor(item.status, item.type)} ${item.type === 'task' ? getPriorityColor(item.priority) : ''
                                                }`}
                                        >
                                            <div className="flex items-center gap-1">
                                                {item.type === 'event' ? (
                                                    <span>{getEventTypeIcon(item.event_type)}</span>
                                                ) : (
                                                    <CheckCircle className="w-3 h-3" />
                                                )}
                                                <span className="truncate">{item.title}</span>
                                            </div>
                                            {item.type === 'event' && item.start_datetime && (
                                                <div className="text-xs text-gray-500">
                                                    {formatTime(item.start_datetime)}
                                                </div>
                                            )}
                                        </div>
                                    ))}
                                </div>
                            </div>
                        );
                    })}
                </div>
            </div>
        );
    };

    // Render day view
    const renderDayView = () => {
        const dayItems = getItemsForDate(currentDate);
        const isToday = currentDate.toDateString() === new Date().toDateString();

        return (
            <div className="bg-white rounded-lg shadow overflow-hidden">
                {/* Header */}
                <div className="flex items-center justify-between p-4 border-b">
                    <h2 className="text-xl font-semibold">
                        {currentDate.toLocaleDateString('he-IL', { weekday: 'long', day: 'numeric', month: 'long', year: 'numeric' })}
                    </h2>
                    <div className="flex items-center gap-2">
                        <button
                            onClick={goToPrevious}
                            className="p-2 hover:bg-gray-100 rounded-md transition-colors"
                            title="קודם"
                            aria-label="קודם"
                        >
                            <ChevronRight className="w-4 h-4" />
                        </button>
                        <button
                            onClick={goToToday}
                            className="px-3 py-1 text-sm bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors"
                        >
                            היום
                        </button>
                        <button
                            onClick={goToNext}
                            className="p-2 hover:bg-gray-100 rounded-md transition-colors"
                            title="הבא"
                            aria-label="הבא"
                        >
                            <ChevronLeft className="w-4 h-4" />
                        </button>
                    </div>
                </div>

                {/* Day content */}
                <div className="p-4">
                    {dayItems.length === 0 ? (
                        <div className="text-center text-gray-500 py-8">
                            אין אירועים או משימות ליום זה
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {dayItems.map(item => (
                                <div
                                    key={`${item.type}-${item.id}`}
                                    className={`p-4 rounded-lg border-l-4 ${getStatusColor(item.status, item.type)} ${item.type === 'task' ? getPriorityColor(item.priority) : ''
                                        }`}
                                >
                                    <div className="flex items-start justify-between">
                                        <div className="flex-1">
                                            <div className="flex items-center gap-2 mb-2">
                                                {item.type === 'event' ? (
                                                    <span className="text-lg">{getEventTypeIcon(item.event_type)}</span>
                                                ) : (
                                                    <CheckCircle className="w-5 h-5" />
                                                )}
                                                <h3 className="font-medium">{item.title}</h3>
                                            </div>
                                            <div className="text-sm text-gray-600">
                                                {item.type === 'event' && item.start_datetime && (
                                                    <div className="flex items-center gap-1">
                                                        <Clock className="w-4 h-4" />
                                                        {formatTime(item.start_datetime)}
                                                    </div>
                                                )}
                                                {item.customer_name && (
                                                    <div className="flex items-center gap-1">
                                                        <Users className="w-4 h-4" />
                                                        {item.customer_name}
                                                    </div>
                                                )}
                                                {item.type === 'event' && item.location && (
                                                    <div className="flex items-center gap-1">
                                                        <MapPin className="w-4 h-4" />
                                                        {item.location}
                                                    </div>
                                                )}
                                            </div>
                                        </div>
                                        <div className="text-sm text-gray-500">
                                            {item.type === 'event' ? 'אירוע' : 'משימה'}
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            </div>
        );
    };

    if (loading) {
        return (
            <div className="bg-white rounded-lg shadow p-8">
                <div className="animate-pulse">
                    <div className="h-6 bg-gray-200 rounded w-1/4 mb-4"></div>
                    <div className="space-y-3">
                        {[1, 2, 3, 4, 5].map(i => (
                            <div key={i} className="h-16 bg-gray-200 rounded"></div>
                        ))}
                    </div>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded">
                {error}
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* View controls */}
            <div className="bg-white p-4 rounded-lg shadow">
                <div className="flex items-center justify-between">
                    <h2 className="text-xl font-semibold">לוח שנה</h2>
                    <div className="flex items-center gap-2">
                        <button
                            onClick={() => setView('month')}
                            className={`px-3 py-1 text-sm rounded-md transition-colors ${view === 'month' ? 'bg-blue-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                }`}
                        >
                            חודש
                        </button>
                        <button
                            onClick={() => setView('week')}
                            className={`px-3 py-1 text-sm rounded-md transition-colors ${view === 'week' ? 'bg-blue-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                }`}
                        >
                            שבוע
                        </button>
                        <button
                            onClick={() => setView('day')}
                            className={`px-3 py-1 text-sm rounded-md transition-colors ${view === 'day' ? 'bg-blue-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                }`}
                        >
                            יום
                        </button>
                    </div>
                </div>
            </div>

            {/* Calendar content */}
            {view === 'month' && renderMonthView()}
            {view === 'week' && renderWeekView()}
            {view === 'day' && renderDayView()}
        </div>
    );
}
