'use client';

import { useState, useEffect } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { useAuthStore } from '@/src/store/authStore';
import Link from 'next/link';
import { Eye, EyeOff, Lock, CheckCircle } from 'lucide-react';

const resetPasswordSchema = z.object({
    password: z.string().min(8, 'הסיסמה חייבת להכיל לפחות 8 תווים').min(1, 'סיסמה נדרשת'),
    confirmPassword: z.string().min(1, 'אישור סיסמה נדרש'),
}).refine((data) => data.password === data.confirmPassword, {
    message: "הסיסמאות אינן תואמות",
    path: ["confirmPassword"],
});

type ResetPasswordForm = z.infer<typeof resetPasswordSchema>;

export default function ResetPasswordPage() {
    const [showPassword, setShowPassword] = useState(false);
    const [showConfirmPassword, setShowConfirmPassword] = useState(false);
    const [isSuccess, setIsSuccess] = useState(false);
    const [token, setToken] = useState<string | null>(null);
    const { resetPassword, isLoading, error, clearError } = useAuthStore();
    const router = useRouter();
    const searchParams = useSearchParams();

    const {
        register,
        handleSubmit,
        formState: { errors },
    } = useForm<ResetPasswordForm>({
        resolver: zodResolver(resetPasswordSchema),
    });

    useEffect(() => {
        const tokenParam = searchParams.get('token');
        if (!tokenParam) {
            router.push('/forgot-password');
            return;
        }
        setToken(tokenParam);
    }, [searchParams, router]);

    useEffect(() => {
        clearError();
    }, [clearError]);

    const onSubmit = async (data: ResetPasswordForm) => {
        if (!token) return;

        await resetPassword(token, data.password);
        setIsSuccess(true);
    };

    if (isSuccess) {
        return (
            <div className="min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
                <div className="max-w-md w-full space-y-8">
                    <div className="text-center">
                        <h1 className="text-3xl font-bold text-gray-900 mb-2">מערכת ניהול דומיינים</h1>
                    </div>

                    <Card className="shadow-lg">
                        <CardHeader className="text-center">
                            <div className="mx-auto w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mb-4">
                                <CheckCircle className="w-6 h-6 text-green-600" />
                            </div>
                            <CardTitle className="text-2xl font-semibold">הסיסמה אופסה בהצלחה</CardTitle>
                            <CardDescription>
                                הסיסמה שלך עודכנה בהצלחה
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="text-center space-y-4">
                            <p className="text-gray-600">
                                כעת תוכל להתחבר עם הסיסמה החדשה שלך.
                            </p>
                            <Link href="/login">
                                <Button className="w-full">
                                    התחבר עכשיו
                                </Button>
                            </Link>
                        </CardContent>
                    </Card>
                </div>
            </div>
        );
    }

    if (!token) {
        return (
            <div className="min-h-screen flex items-center justify-center bg-gray-50">
                <div className="text-center">
                    <h1 className="text-2xl font-bold text-gray-900">טוען...</h1>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
            <div className="max-w-md w-full space-y-8">
                <div className="text-center">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">מערכת ניהול דומיינים</h1>
                    <p className="text-gray-600">איפוס סיסמה</p>
                </div>

                <Card className="shadow-lg">
                    <CardHeader className="text-center">
                        <CardTitle className="text-2xl font-semibold">סיסמה חדשה</CardTitle>
                        <CardDescription>
                            הכנס סיסמה חדשה למערכת
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={handleSubmit(onSubmit)} className="space-y-6">
                            {error && (
                                <div className="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-md text-sm">
                                    {error}
                                </div>
                            )}

                            <div className="space-y-2">
                                <Label htmlFor="password">סיסמה חדשה</Label>
                                <div className="relative">
                                    <Lock className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="password"
                                        type={showPassword ? 'text' : 'password'}
                                        placeholder="הכנס סיסמה חדשה"
                                        className="pr-10"
                                        {...register('password')}
                                    />
                                    <button
                                        type="button"
                                        className="absolute left-3 top-3 text-gray-400 hover:text-gray-600"
                                        onClick={() => setShowPassword(!showPassword)}
                                    >
                                        {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                                    </button>
                                </div>
                                {errors.password && (
                                    <p className="text-sm text-red-600">{errors.password.message}</p>
                                )}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="confirmPassword">אישור סיסמה</Label>
                                <div className="relative">
                                    <Lock className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="confirmPassword"
                                        type={showConfirmPassword ? 'text' : 'password'}
                                        placeholder="הכנס סיסמה שוב"
                                        className="pr-10"
                                        {...register('confirmPassword')}
                                    />
                                    <button
                                        type="button"
                                        className="absolute left-3 top-3 text-gray-400 hover:text-gray-600"
                                        onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                                    >
                                        {showConfirmPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                                    </button>
                                </div>
                                {errors.confirmPassword && (
                                    <p className="text-sm text-red-600">{errors.confirmPassword.message}</p>
                                )}
                            </div>

                            <div className="bg-blue-50 p-4 rounded-md">
                                <p className="text-sm text-blue-800 font-medium mb-2">דרישות סיסמה:</p>
                                <ul className="text-xs text-blue-700 space-y-1">
                                    <li>• לפחות 8 תווים</li>
                                    <li>• אות קטנה אחת לפחות</li>
                                    <li>• אות גדולה אחת לפחות</li>
                                    <li>• ספרה אחת לפחות</li>
                                </ul>
                            </div>

                            <Button
                                type="submit"
                                className="w-full"
                                disabled={isLoading}
                            >
                                {isLoading ? 'מאפס סיסמה...' : 'אפס סיסמה'}
                            </Button>
                        </form>

                        <div className="mt-6 text-center">
                            <Link
                                href="/login"
                                className="text-sm text-blue-600 hover:text-blue-500 hover:underline"
                            >
                                חזור להתחברות
                            </Link>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </div>
    );
}
