'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { useAuthStore } from '@/src/store/authStore';
import Link from 'next/link';
import { Eye, EyeOff, Mail, Lock, User } from 'lucide-react';

const registerSchema = z.object({
    name: z.string().min(2, 'השם חייב להכיל לפחות 2 תווים').min(1, 'השם נדרש'),
    email: z.string().email('כתובת אימייל לא תקינה').min(1, 'כתובת אימייל נדרשת'),
    password: z.string().min(8, 'הסיסמה חייבת להכיל לפחות 8 תווים').min(1, 'סיסמה נדרשת'),
    confirmPassword: z.string().min(1, 'אישור סיסמה נדרש'),
}).refine((data) => data.password === data.confirmPassword, {
    message: "הסיסמאות אינן תואמות",
    path: ["confirmPassword"],
});

type RegisterForm = z.infer<typeof registerSchema>;

export default function RegisterPage() {
    const [showPassword, setShowPassword] = useState(false);
    const [showConfirmPassword, setShowConfirmPassword] = useState(false);
    const { register: registerUser, isLoading, error, clearError, isAuthenticated } = useAuthStore();
    const router = useRouter();

    const {
        register,
        handleSubmit,
        formState: { errors },
    } = useForm<RegisterForm>({
        resolver: zodResolver(registerSchema),
    });

    useEffect(() => {
        if (isAuthenticated) {
            router.push('/');
        }
    }, [isAuthenticated, router]);

    useEffect(() => {
        clearError();
    }, [clearError]);

    const onSubmit = async (data: RegisterForm) => {
        await registerUser({
            email: data.email,
            password: data.password,
            name: data.name,
            role: 'USER',
        });
    };

    return (
        <div className="min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
            <div className="max-w-md w-full space-y-8">
                <div className="text-center">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">מערכת ניהול דומיינים</h1>
                    <p className="text-gray-600">הירשם למערכת</p>
                </div>

                <Card className="shadow-lg">
                    <CardHeader className="text-center">
                        <CardTitle className="text-2xl font-semibold">הרשמה</CardTitle>
                        <CardDescription>
                            צור חשבון חדש
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={handleSubmit(onSubmit)} className="space-y-6">
                            {error && (
                                <div className="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-md text-sm">
                                    {error}
                                </div>
                            )}

                            <div className="space-y-2">
                                <Label htmlFor="name">שם מלא</Label>
                                <div className="relative">
                                    <User className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="name"
                                        type="text"
                                        placeholder="הכנס שם מלא"
                                        className="pr-10"
                                        {...register('name')}
                                    />
                                </div>
                                {errors.name && (
                                    <p className="text-sm text-red-600">{errors.name.message}</p>
                                )}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="email">כתובת אימייל</Label>
                                <div className="relative">
                                    <Mail className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="email"
                                        type="email"
                                        placeholder="הכנס כתובת אימייל"
                                        className="pr-10"
                                        {...register('email')}
                                    />
                                </div>
                                {errors.email && (
                                    <p className="text-sm text-red-600">{errors.email.message}</p>
                                )}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="password">סיסמה</Label>
                                <div className="relative">
                                    <Lock className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="password"
                                        type={showPassword ? 'text' : 'password'}
                                        placeholder="הכנס סיסמה"
                                        className="pr-10"
                                        {...register('password')}
                                    />
                                    <button
                                        type="button"
                                        className="absolute left-3 top-3 text-gray-400 hover:text-gray-600"
                                        onClick={() => setShowPassword(!showPassword)}
                                    >
                                        {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                                    </button>
                                </div>
                                {errors.password && (
                                    <p className="text-sm text-red-600">{errors.password.message}</p>
                                )}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="confirmPassword">אישור סיסמה</Label>
                                <div className="relative">
                                    <Lock className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="confirmPassword"
                                        type={showConfirmPassword ? 'text' : 'password'}
                                        placeholder="הכנס סיסמה שוב"
                                        className="pr-10"
                                        {...register('confirmPassword')}
                                    />
                                    <button
                                        type="button"
                                        className="absolute left-3 top-3 text-gray-400 hover:text-gray-600"
                                        onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                                    >
                                        {showConfirmPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                                    </button>
                                </div>
                                {errors.confirmPassword && (
                                    <p className="text-sm text-red-600">{errors.confirmPassword.message}</p>
                                )}
                            </div>

                            <div className="bg-blue-50 p-4 rounded-md">
                                <p className="text-sm text-blue-800 font-medium mb-2">דרישות סיסמה:</p>
                                <ul className="text-xs text-blue-700 space-y-1">
                                    <li>• לפחות 8 תווים</li>
                                    <li>• אות קטנה אחת לפחות</li>
                                    <li>• אות גדולה אחת לפחות</li>
                                    <li>• ספרה אחת לפחות</li>
                                </ul>
                            </div>

                            <Button
                                type="submit"
                                className="w-full"
                                disabled={isLoading}
                            >
                                {isLoading ? 'נרשם...' : 'הירשם'}
                            </Button>
                        </form>

                        <div className="mt-6 text-center">
                            <div className="text-sm text-gray-600">
                                כבר יש לך חשבון?{' '}
                                <Link
                                    href="/login"
                                    className="text-blue-600 hover:text-blue-500 hover:underline"
                                >
                                    התחבר כאן
                                </Link>
                            </div>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </div>
    );
}
