'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Plus, Search, Filter, Download, Tag } from 'lucide-react';
import Sidebar from '@/src/components/layout/Sidebar';
import Header from '@/src/components/layout/Header';
import { ProjectDrawer } from '@/components/projects/ProjectDrawer';

interface Project {
    id: number;
    title: string;
    description?: string;
    customer_name: string;
    customer_company?: string;
    domain_name?: string;
    project_type: string;
    status: string;
    start_date?: string;
    due_date?: string;
    budget: number;
    progress: number;
    is_open: boolean;
    priority: string;
    tags: string[];
    created_date: string;
}

interface ProjectTag {
    id: number;
    name: string;
    color: string;
    description?: string;
}

export default function ProjectsPage() {
    const [projects, setProjects] = useState<Project[]>([]);
    const [projectTags, setProjectTags] = useState<ProjectTag[]>([]);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');
    const [statusFilter, setStatusFilter] = useState('all');
    const [typeFilter, setTypeFilter] = useState('all');
    const [openFilter, setOpenFilter] = useState('all');
    const [drawerOpen, setDrawerOpen] = useState(false);
    const [selectedProject, setSelectedProject] = useState<Project | null>(null);

    // טעינת פרויקטים
    const loadProjects = async () => {
        try {
            setLoading(true);
            const params = new URLSearchParams();
            if (searchTerm) params.append('search', searchTerm);
            if (statusFilter && statusFilter !== 'all') params.append('status', statusFilter);
            if (typeFilter && typeFilter !== 'all') params.append('project_type', typeFilter);
            if (openFilter && openFilter !== 'all') params.append('is_open', openFilter);

            const response = await fetch(`/api/projects?${params.toString()}`);
            if (response.ok) {
                const data = await response.json();
                setProjects(data.projects || []);
            }
        } catch (error) {
            console.error('Error loading projects:', error);
        } finally {
            setLoading(false);
        }
    };

    // טעינת תגיות
    const loadProjectTags = async () => {
        try {
            const response = await fetch('/api/project-tags');
            if (response.ok) {
                const tags = await response.json();
                setProjectTags(tags);
            }
        } catch (error) {
            console.error('Error loading project tags:', error);
        }
    };

    useEffect(() => {
        loadProjects();
        loadProjectTags();
    }, [searchTerm, statusFilter, typeFilter, openFilter]);

    const getStatusColor = (status: string) => {
        const colors: Record<string, string> = {
            open: 'bg-green-100 text-green-800',
            in_progress: 'bg-blue-100 text-blue-800',
            pending: 'bg-yellow-100 text-yellow-800',
            completed: 'bg-gray-100 text-gray-800',
            cancelled: 'bg-red-100 text-red-800',
        };
        return colors[status] || 'bg-gray-100 text-gray-800';
    };

    const getStatusLabel = (status: string) => {
        const labels: Record<string, string> = {
            open: 'פתוח',
            in_progress: 'בביצוע',
            pending: 'ממתין',
            completed: 'הושלם',
            cancelled: 'בוטל',
        };
        return labels[status] || status;
    };

    const getProjectTypeLabel = (type: string) => {
        const labels: Record<string, string> = {
            website_design: 'עיצוב אתר',
            website_development: 'פיתוח אתר',
            domain_setup: 'הגדרת דומיין',
            email_setup: 'הגדרת אימייל',
            maintenance: 'תחזוקה',
            hosting_migration: 'העברת אחסון',
            seo: 'SEO',
            marketing: 'שיווק',
            other: 'אחר',
        };
        return labels[type] || type;
    };

    const getPriorityColor = (priority: string) => {
        const colors: Record<string, string> = {
            urgent: 'bg-red-100 text-red-800',
            high: 'bg-orange-100 text-orange-800',
            medium: 'bg-yellow-100 text-yellow-800',
            low: 'bg-green-100 text-green-800',
        };
        return colors[priority] || 'bg-gray-100 text-gray-800';
    };

    const getPriorityLabel = (priority: string) => {
        const labels: Record<string, string> = {
            urgent: 'דחוף',
            high: 'גבוה',
            medium: 'בינוני',
            low: 'נמוך',
        };
        return labels[priority] || priority;
    };

    const formatDate = (dateString: string) => {
        if (!dateString) return 'לא זמין';
        const date = new Date(dateString);
        return date.toLocaleDateString('he-IL');
    };

    const formatCurrency = (amount: number) => {
        if (!amount) return 'לא צוין';
        return `₪${amount.toLocaleString('he-IL')}`;
    };

    const handleAddProject = () => {
        setSelectedProject(null);
        setDrawerOpen(true);
    };

    const handleEditProject = (project: Project) => {
        // מעבר לדף עריכת הפרויקט
        window.location.href = `/projects/${project.id}/edit`;
    };

    const handleDrawerSuccess = () => {
        loadProjects();
    };

    const handleViewProject = (project: Project) => {
        // מעבר לדף הפרויקט הנפרד
        window.location.href = `/projects/${project.id}`;
    };

    return (
        <div className="min-h-screen bg-gray-50 flex">
            <Sidebar />
            <div className="flex-1 flex flex-col">
                <Header />
                <div className="flex-1 p-6" dir="rtl">
                    {/* כותרת וכפתורים */}
                    <div className="flex items-center justify-between mb-6">
                        <div>
                            <h1 className="text-2xl font-bold">📊 פרויקטים</h1>
                            <p className="text-gray-600 mt-1">ניהול פרויקטים ועקבות התקדמות</p>
                        </div>

                        <div className="flex gap-3">
                            <Button
                                className="bg-blue-500 hover:bg-blue-600"
                                onClick={handleAddProject}
                            >
                                <Plus className="w-4 h-4 ml-2" />
                                הוסף פרויקט
                            </Button>

                            <Button variant="outline">
                                <Download className="w-4 h-4 ml-2" />
                                Import
                            </Button>

                            <Button variant="outline">
                                <Tag className="w-4 h-4 ml-2" />
                                ניהול תוויות
                            </Button>
                        </div>
                    </div>

                    {/* פילטרים */}
                    <Card className="mb-6">
                        <CardHeader>
                            <CardTitle className="flex items-center gap-2">
                                <Filter className="w-5 h-5" />
                                פילטרים
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                                <div>
                                    <label className="block text-sm font-medium mb-2">חיפוש</label>
                                    <div className="relative">
                                        <Search className="absolute right-3 top-2.5 h-4 w-4 text-gray-400" />
                                        <Input
                                            placeholder="חיפוש בפרויקטים..."
                                            value={searchTerm}
                                            onChange={(e) => setSearchTerm(e.target.value)}
                                            className="pr-10"
                                        />
                                    </div>
                                </div>

                                <div>
                                    <label className="block text-sm font-medium mb-2">סטטוס</label>
                                    <Select value={statusFilter} onValueChange={setStatusFilter}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="כל הסטטוסים" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="all">כל הסטטוסים</SelectItem>
                                            <SelectItem value="open">פתוח</SelectItem>
                                            <SelectItem value="in_progress">בביצוע</SelectItem>
                                            <SelectItem value="pending">ממתין</SelectItem>
                                            <SelectItem value="completed">הושלם</SelectItem>
                                            <SelectItem value="cancelled">בוטל</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>

                                <div>
                                    <label className="block text-sm font-medium mb-2">סוג פרויקט</label>
                                    <Select value={typeFilter} onValueChange={setTypeFilter}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="כל הסוגים" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="all">כל הסוגים</SelectItem>
                                            <SelectItem value="website_design">עיצוב אתר</SelectItem>
                                            <SelectItem value="website_development">פיתוח אתר</SelectItem>
                                            <SelectItem value="domain_setup">הגדרת דומיין</SelectItem>
                                            <SelectItem value="email_setup">הגדרת אימייל</SelectItem>
                                            <SelectItem value="maintenance">תחזוקה</SelectItem>
                                            <SelectItem value="hosting_migration">העברת אחסון</SelectItem>
                                            <SelectItem value="seo">SEO</SelectItem>
                                            <SelectItem value="marketing">שיווק</SelectItem>
                                            <SelectItem value="other">אחר</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>

                                <div>
                                    <label className="block text-sm font-medium mb-2">פרויקט פתוח</label>
                                    <Select value={openFilter} onValueChange={setOpenFilter}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="כל הפרויקטים" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="all">כל הפרויקטים</SelectItem>
                                            <SelectItem value="true">פתוחים בלבד</SelectItem>
                                            <SelectItem value="false">סגורים בלבד</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    {/* טבלת פרויקטים */}
                    <Card>
                        <CardHeader>
                            <CardTitle>📋 רשימת פרויקטים</CardTitle>
                        </CardHeader>
                        <CardContent>
                            {loading ? (
                                <div className="text-center py-8">טוען פרויקטים...</div>
                            ) : projects.length === 0 ? (
                                <div className="text-center py-8 text-gray-500">
                                    לא נמצאו פרויקטים
                                </div>
                            ) : (
                                <div className="overflow-x-auto">
                                    <table className="w-full">
                                        <thead>
                                            <tr className="border-b">
                                                <th className="text-right p-3 font-medium">#</th>
                                                <th className="text-right p-3 font-medium">שם הפרויקט</th>
                                                <th className="text-right p-3 font-medium">לקוח</th>
                                                <th className="text-right p-3 font-medium">דומיין</th>
                                                <th className="text-right p-3 font-medium">סוג</th>
                                                <th className="text-right p-3 font-medium">תאריך התחלה</th>
                                                <th className="text-right p-3 font-medium">מועד אחרון</th>
                                                <th className="text-right p-3 font-medium">תקציב</th>
                                                <th className="text-right p-3 font-medium">סטטוס</th>
                                                <th className="text-right p-3 font-medium">התקדמות</th>
                                                <th className="text-right p-3 font-medium">פעולות</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            {projects.map((project) => (
                                                <tr key={project.id} className="border-b hover:bg-gray-50">
                                                    <td className="p-3 text-gray-500">#{project.id}</td>

                                                    <td className="p-3">
                                                        <div className="font-medium">{project.title}</div>
                                                        {project.is_open && (
                                                            <Badge variant="outline" className="mt-1 text-xs">
                                                                פתוח
                                                            </Badge>
                                                        )}
                                                    </td>

                                                    <td className="p-3">
                                                        <a
                                                            href={`/customers?search=${encodeURIComponent(project.customer_name)}`}
                                                            className="font-medium text-blue-600 hover:text-blue-800 hover:underline"
                                                        >
                                                            {project.customer_name}
                                                        </a>
                                                        {project.customer_company && (
                                                            <div className="text-sm text-gray-600">{project.customer_company}</div>
                                                        )}
                                                    </td>

                                                    <td className="p-3">
                                                        {project.domain_name && (
                                                            <span className="text-sm text-blue-600">🌐 {project.domain_name}</span>
                                                        )}
                                                    </td>

                                                    <td className="p-3">
                                                        <Badge variant="secondary">
                                                            {getProjectTypeLabel(project.project_type)}
                                                        </Badge>
                                                    </td>

                                                    <td className="p-3 text-sm">
                                                        {formatDate(project.start_date)}
                                                    </td>

                                                    <td className="p-3 text-sm">
                                                        {formatDate(project.due_date)}
                                                    </td>

                                                    <td className="p-3 text-sm">
                                                        {formatCurrency(project.budget)}
                                                    </td>

                                                    <td className="p-3">
                                                        <Badge className={getStatusColor(project.status)}>
                                                            {getStatusLabel(project.status)}
                                                        </Badge>
                                                    </td>

                                                    <td className="p-3">
                                                        <div className="flex items-center gap-2">
                                                            <div className="w-20 bg-gray-200 rounded-full h-2">
                                                                <div
                                                                    className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                                                    style={{ width: `${project.progress}%` }}
                                                                ></div>
                                                            </div>
                                                            <span className="text-sm text-gray-600">{project.progress}%</span>
                                                        </div>
                                                    </td>

                                                    <td className="p-3">
                                                        <div className="flex gap-2">
                                                            <Button
                                                                size="sm"
                                                                variant="outline"
                                                                onClick={() => handleEditProject(project)}
                                                            >
                                                                ערוך
                                                            </Button>
                                                            <Button
                                                                size="sm"
                                                                variant="outline"
                                                                onClick={() => handleViewProject(project)}
                                                            >
                                                                צפה
                                                            </Button>
                                                        </div>
                                                    </td>
                                                </tr>
                                            ))}
                                        </tbody>
                                    </table>
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    {/* ProjectDrawer */}
                    <ProjectDrawer
                        project={selectedProject}
                        open={drawerOpen}
                        onOpenChange={setDrawerOpen}
                        onSuccess={handleDrawerSuccess}
                        isViewMode={selectedProject?.isViewMode || false}
                    />
                </div>
            </div>
        </div>
    );
}
