'use client';

import { useState, useEffect } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { useAuthStore } from '@/src/store/authStore';
import Link from 'next/link';
import { Eye, EyeOff, Mail, Lock } from 'lucide-react';

const loginSchema = z.object({
    email: z.string().email('כתובת אימייל לא תקינה').min(1, 'כתובת אימייל נדרשת'),
    password: z.string().min(1, 'סיסמה נדרשת'),
});

type LoginForm = z.infer<typeof loginSchema>;

export default function LoginPage() {
    const [showPassword, setShowPassword] = useState(false);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const { login, isLoading, error: authError, clearError, isAuthenticated } = useAuthStore();
    const router = useRouter();
    const searchParams = useSearchParams();

    const {
        register,
        handleSubmit,
        formState: { errors },
    } = useForm<LoginForm>({
        resolver: zodResolver(loginSchema),
    });

    const nextUrl = searchParams.get('next') || '/';

    useEffect(() => {
        if (isAuthenticated) {
            router.push(nextUrl);
        }
    }, [isAuthenticated, router, nextUrl]);

    useEffect(() => {
        clearError();
        setError(null);
    }, [clearError]);

    const onSubmit = async (data: LoginForm) => {
        setIsSubmitting(true);

        try {
            const response = await fetch('/api/auth/login', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email: data.email, password: data.password }),
            });

            const result = await response.json();

            if (response.ok && result.user) {
                // הצלחה - מעבר לדף הבית
                window.location.href = nextUrl;
            } else {
                // שגיאה - הצגת הודעת שגיאה
                setError(result.error || 'שגיאה בהתחברות');
                setIsSubmitting(false);
            }
        } catch (error) {
            setError('שגיאת רשת - אנא נסה שוב');
            setIsSubmitting(false);
        }
    };

    return (
        <div className="min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
            <div className="max-w-md w-full space-y-8">
                <div className="text-center">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">מערכת ניהול דומיינים</h1>
                    <p className="text-gray-600">התחבר למערכת</p>
                </div>

                <Card className="shadow-lg">
                    <CardHeader className="text-center">
                        <CardTitle className="text-2xl font-semibold">התחברות</CardTitle>
                        <CardDescription>
                            הכנס את פרטי ההתחברות שלך
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={(e) => { console.log('📝 Form submitted!'); handleSubmit(onSubmit)(e); }} className="space-y-6">
                            {error && (
                                <div className="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-md text-sm">
                                    {error}
                                </div>
                            )}

                            <div className="space-y-2">
                                <Label htmlFor="email">כתובת אימייל</Label>
                                <div className="relative">
                                    <Mail className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="email"
                                        type="email"
                                        placeholder="הכנס כתובת אימייל"
                                        className="pr-10"
                                        {...register('email')}
                                    />
                                </div>
                                {errors.email && (
                                    <p className="text-sm text-red-600">{errors.email.message}</p>
                                )}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="password">סיסמה</Label>
                                <div className="relative">
                                    <Lock className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="password"
                                        type={showPassword ? 'text' : 'password'}
                                        placeholder="הכנס סיסמה"
                                        className="pr-10"
                                        {...register('password')}
                                    />
                                    <button
                                        type="button"
                                        className="absolute left-3 top-3 text-gray-400 hover:text-gray-600"
                                        onClick={() => setShowPassword(!showPassword)}
                                    >
                                        {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                                    </button>
                                </div>
                                {errors.password && (
                                    <p className="text-sm text-red-600">{errors.password.message}</p>
                                )}
                            </div>

                            <Button
                                type="submit"
                                className="w-full"
                                disabled={isSubmitting}
                            >
                                {isSubmitting ? 'מתחבר...' : 'התחבר'}
                            </Button>
                        </form>

                        <div className="mt-6 text-center space-y-2">
                            <Link
                                href="/forgot-password"
                                className="text-sm text-blue-600 hover:text-blue-500 hover:underline"
                            >
                                שכחתי את הסיסמה
                            </Link>
                            <div className="text-sm text-gray-600">
                                אין לך חשבון?{' '}
                                <Link
                                    href="/register"
                                    className="text-blue-600 hover:text-blue-500 hover:underline"
                                >
                                    הירשם כאן
                                </Link>
                            </div>
                        </div>

                        <div className="mt-6 p-4 bg-gray-50 rounded-md">
                            <p className="text-sm text-gray-600 mb-2">משתמש ברירת מחדל:</p>
                            <p className="text-xs text-gray-500">
                                אימייל: admin@domain-manager.com<br />
                                סיסמה: admin123
                            </p>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </div>
    );
}
