'use client';

import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { useAuthStore } from '@/src/store/authStore';
import Link from 'next/link';
import { Mail, ArrowRight } from 'lucide-react';

const forgotPasswordSchema = z.object({
    email: z.string().email('כתובת אימייל לא תקינה').min(1, 'כתובת אימייל נדרשת'),
});

type ForgotPasswordForm = z.infer<typeof forgotPasswordSchema>;

export default function ForgotPasswordPage() {
    const [isSubmitted, setIsSubmitted] = useState(false);
    const { forgotPassword, isLoading, error, clearError } = useAuthStore();

    const {
        register,
        handleSubmit,
        formState: { errors },
    } = useForm<ForgotPasswordForm>({
        resolver: zodResolver(forgotPasswordSchema),
    });

    const onSubmit = async (data: ForgotPasswordForm) => {
        await forgotPassword(data.email);
        setIsSubmitted(true);
    };

    if (isSubmitted) {
        return (
            <div className="min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
                <div className="max-w-md w-full space-y-8">
                    <div className="text-center">
                        <h1 className="text-3xl font-bold text-gray-900 mb-2">מערכת ניהול דומיינים</h1>
                    </div>

                    <Card className="shadow-lg">
                        <CardHeader className="text-center">
                            <div className="mx-auto w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mb-4">
                                <Mail className="w-6 h-6 text-green-600" />
                            </div>
                            <CardTitle className="text-2xl font-semibold">אימייל נשלח</CardTitle>
                            <CardDescription>
                                שלחנו לך קישור לאיפוס הסיסמה
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="text-center space-y-4">
                            <p className="text-gray-600">
                                בדוק את תיבת הדואר שלך ולחץ על הקישור שנשלח כדי לאפס את הסיסמה.
                            </p>
                            <p className="text-sm text-gray-500">
                                לא קיבלת אימייל? בדוק גם בתיקיית הספאם או נסה שוב.
                            </p>
                            <div className="space-y-2">
                                <Button
                                    onClick={() => setIsSubmitted(false)}
                                    variant="outline"
                                    className="w-full"
                                >
                                    שלח שוב
                                </Button>
                                <Link href="/login">
                                    <Button variant="ghost" className="w-full">
                                        חזור להתחברות
                                    </Button>
                                </Link>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
            <div className="max-w-md w-full space-y-8">
                <div className="text-center">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">מערכת ניהול דומיינים</h1>
                    <p className="text-gray-600">שחזור סיסמה</p>
                </div>

                <Card className="shadow-lg">
                    <CardHeader className="text-center">
                        <CardTitle className="text-2xl font-semibold">שכחתי את הסיסמה</CardTitle>
                        <CardDescription>
                            הכנס את כתובת האימייל שלך ונשלח לך קישור לאיפוס הסיסמה
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={handleSubmit(onSubmit)} className="space-y-6">
                            {error && (
                                <div className="bg-red-50 border border-red-200 text-red-600 px-4 py-3 rounded-md text-sm">
                                    {error}
                                </div>
                            )}

                            <div className="space-y-2">
                                <Label htmlFor="email">כתובת אימייל</Label>
                                <div className="relative">
                                    <Mail className="absolute right-3 top-3 h-4 w-4 text-gray-400" />
                                    <Input
                                        id="email"
                                        type="email"
                                        placeholder="הכנס כתובת אימייל"
                                        className="pr-10"
                                        {...register('email')}
                                    />
                                </div>
                                {errors.email && (
                                    <p className="text-sm text-red-600">{errors.email.message}</p>
                                )}
                            </div>

                            <Button
                                type="submit"
                                className="w-full"
                                disabled={isLoading}
                            >
                                {isLoading ? 'שולח...' : 'שלח קישור איפוס'}
                                <ArrowRight className="w-4 h-4 mr-2" />
                            </Button>
                        </form>

                        <div className="mt-6 text-center">
                            <Link
                                href="/login"
                                className="text-sm text-blue-600 hover:text-blue-500 hover:underline"
                            >
                                חזור להתחברות
                            </Link>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </div>
    );
}
