'use client';

import { useState } from 'react';
import AppLayout from '@/src/components/layout/AppLayout';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { TestTube, CheckCircle, XCircle, Loader2 } from 'lucide-react';

export default function TestSMTPPage() {
    const [isTesting, setIsTesting] = useState(false);
    const [testResult, setTestResult] = useState<{
        success: boolean;
        message?: string;
        error?: string;
    } | null>(null);

    const testConnection = async () => {
        setIsTesting(true);
        setTestResult(null);

        try {
            const response = await fetch('/api/emails/send');
            const result = await response.json();

            setTestResult(result);
        } catch (error) {
            setTestResult({
                success: false,
                error: error instanceof Error ? error.message : 'Unknown error'
            });
        } finally {
            setIsTesting(false);
        }
    };

    return (
        <AppLayout>
            <div className="p-6">
                <div className="max-w-4xl mx-auto">
                    {/* Header */}
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900 mb-2">בדיקת חיבור SMTP</h1>
                        <p className="text-gray-600">בדוק את חיבור ה-SMTP שלך לפני שליחת אימיילים</p>
                    </div>

                    {/* Test Card */}
                    <Card className="p-8">
                        <div className="text-center">
                            <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                <TestTube className="w-8 h-8 text-blue-600" />
                            </div>

                            <h2 className="text-2xl font-semibold text-gray-900 mb-4">
                                בדיקת חיבור SMTP
                            </h2>

                            <p className="text-gray-600 mb-8 max-w-2xl mx-auto">
                                לחץ על הכפתור למטה כדי לבדוק את חיבור ה-SMTP שלך.
                                הבדיקה תאמת שהגדרות השרת שלך נכונות ושאתה יכול לשלוח אימיילים.
                            </p>

                            <Button
                                onClick={testConnection}
                                disabled={isTesting}
                                className="bg-blue-600 hover:bg-blue-700 text-white px-8 py-3 text-lg"
                            >
                                {isTesting ? (
                                    <>
                                        <Loader2 className="w-5 h-5 ml-2 animate-spin" />
                                        בודק חיבור...
                                    </>
                                ) : (
                                    <>
                                        <TestTube className="w-5 h-5 ml-2" />
                                        בדוק חיבור SMTP
                                    </>
                                )}
                            </Button>

                            {/* Test Result */}
                            {testResult && (
                                <div className="mt-8">
                                    <Card className={`p-6 ${testResult.success ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200'}`}>
                                        <div className="flex items-center justify-center mb-4">
                                            {testResult.success ? (
                                                <CheckCircle className="w-8 h-8 text-green-600" />
                                            ) : (
                                                <XCircle className="w-8 h-8 text-red-600" />
                                            )}
                                        </div>

                                        <h3 className={`text-xl font-semibold mb-2 ${testResult.success ? 'text-green-800' : 'text-red-800'}`}>
                                            {testResult.success ? 'החיבור הצליח!' : 'החיבור נכשל'}
                                        </h3>

                                        <p className={`text-lg ${testResult.success ? 'text-green-700' : 'text-red-700'}`}>
                                            {testResult.success ? testResult.message : testResult.error}
                                        </p>
                                    </Card>
                                </div>
                            )}

                            {/* Configuration Help */}
                            <div className="mt-12 text-right">
                                <h3 className="text-lg font-semibold text-gray-900 mb-4">הגדרות SMTP נדרשות</h3>
                                <div className="bg-gray-100 rounded-lg p-6 text-sm">
                                    <p className="text-gray-700 mb-2">
                                        כדי לבדוק את החיבור, וודא שהגדרת את המשתנים הבאים בקובץ .env.local:
                                    </p>
                                    <div className="bg-gray-800 text-green-400 p-4 rounded mt-4 font-mono text-xs">
                                        <div>SMTP_HOST=mail.yourdomain.com</div>
                                        <div>SMTP_PORT=465</div>
                                        <div>SMTP_SECURE=true</div>
                                        <div>SMTP_USER=noreply@yourdomain.com</div>
                                        <div>SMTP_PASSWORD=your_password_here</div>
                                        <div>SMTP_FROM_NAME=Your Company Name</div>
                                        <div>SMTP_FROM_EMAIL=noreply@yourdomain.com</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </Card>
                </div>
            </div>
        </AppLayout>
    );
}
