'use client';

import { useState } from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Switch } from '../../../src/components/ui/switch';
import AppLayout from '@/src/components/layout/AppLayout';
import {
    Settings,
    Save,
    TestTube,
    CheckCircle,
    XCircle,
    Loader2,
    Eye,
    EyeOff,
    AlertCircle,
    Info
} from 'lucide-react';

export default function SMTPSettingsPage() {
    const [isEditing, setIsEditing] = useState(false);
    const [isTesting, setIsTesting] = useState(false);
    const [testResult, setTestResult] = useState<{
        success: boolean;
        message?: string;
        error?: string;
    } | null>(null);
    const [showPassword, setShowPassword] = useState(false);

    const [smtpSettings, setSmtpSettings] = useState({
        host: 'mail.tsm.co.il',
        port: '465',
        secure: true,
        user: 'sales@tsm.co.il',
        password: '3qewradsf@',
        fromName: 'TSM Company',
        fromEmail: 'sales@tsm.co.il'
    });

    const handleSave = () => {
        // כאן תוכל לשמור את ההגדרות לקובץ .env.local או למסד נתונים
        console.log('Saving SMTP settings:', smtpSettings);
        setIsEditing(false);
        // הודעת הצלחה
        alert('הגדרות SMTP נשמרו בהצלחה!');
    };

    const testConnection = async (customUser?: string, customPass?: string, customPort?: string) => {
        setIsTesting(true);
        setTestResult(null);

        try {
            const params = new URLSearchParams();
            if (customUser && customPass) {
                params.append('testUser', customUser);
                params.append('testPass', customPass);
                if (customPort) params.append('testPort', customPort);
            }

            const response = await fetch(`/api/emails/send${params.toString() ? '?' + params.toString() : ''}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                },
            });
            const result = await response.json();

            setTestResult(result);
        } catch (error) {
            setTestResult({
                success: false,
                error: error instanceof Error ? error.message : 'Unknown error'
            });
        } finally {
            setIsTesting(false);
        }
    };

    const quickTest = async (username: string, portNumber: string) => {
        await testConnection(username, smtpSettings.password, portNumber);
    };

    return (
        <AppLayout>
            <div className="max-w-4xl mx-auto">
                {/* Header */}
                <div className="mb-8">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">הגדרות SMTP</h1>
                    <p className="text-gray-600">הגדר את חיבור ה-SMTP שלך לשליחת אימיילים</p>
                </div>

                {/* Settings Card */}
                <Card className="p-8">
                    <div className="flex justify-between items-center mb-6">
                        <h2 className="text-xl font-semibold text-gray-900">הגדרות שרת SMTP</h2>
                        <div className="flex gap-3">
                            {!isEditing ? (
                                <Button
                                    onClick={() => setIsEditing(true)}
                                    className="bg-blue-600 hover:bg-blue-700"
                                >
                                    <Settings className="w-4 h-4 ml-2" />
                                    ערוך הגדרות
                                </Button>
                            ) : (
                                <div className="flex gap-2">
                                    <Button
                                        onClick={handleSave}
                                        className="bg-green-600 hover:bg-green-700"
                                    >
                                        <Save className="w-4 h-4 ml-2" />
                                        שמור
                                    </Button>
                                    <Button
                                        onClick={() => setIsEditing(false)}
                                        variant="outline"
                                    >
                                        ביטול
                                    </Button>
                                </div>
                            )}
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        {/* Host */}
                        <div className="space-y-2">
                            <Label htmlFor="host">שרת SMTP</Label>
                            <Input
                                id="host"
                                value={smtpSettings.host}
                                onChange={(e) => setSmtpSettings({ ...smtpSettings, host: e.target.value })}
                                disabled={!isEditing}
                                placeholder="mail.yourdomain.com"
                            />
                        </div>

                        {/* Port */}
                        <div className="space-y-2">
                            <Label htmlFor="port">פורט</Label>
                            <Input
                                id="port"
                                value={smtpSettings.port}
                                onChange={(e) => setSmtpSettings({ ...smtpSettings, port: e.target.value })}
                                disabled={!isEditing}
                                placeholder="465"
                            />
                        </div>

                        {/* Username */}
                        <div className="space-y-2">
                            <Label htmlFor="user">שם משתמש</Label>
                            <Input
                                id="user"
                                value={smtpSettings.user}
                                onChange={(e) => setSmtpSettings({ ...smtpSettings, user: e.target.value })}
                                disabled={!isEditing}
                                placeholder="noreply@yourdomain.com"
                            />
                        </div>

                        {/* Password */}
                        <div className="space-y-2">
                            <Label htmlFor="password">סיסמה</Label>
                            <div className="relative">
                                <Input
                                    id="password"
                                    type={showPassword ? "text" : "password"}
                                    value={smtpSettings.password}
                                    onChange={(e) => setSmtpSettings({ ...smtpSettings, password: e.target.value })}
                                    disabled={!isEditing}
                                    placeholder="your_password_here"
                                />
                                <button
                                    type="button"
                                    onClick={() => setShowPassword(!showPassword)}
                                    className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700"
                                    disabled={!isEditing}
                                >
                                    {showPassword ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                                </button>
                            </div>
                        </div>

                        {/* From Name */}
                        <div className="space-y-2">
                            <Label htmlFor="fromName">שם השולח</Label>
                            <Input
                                id="fromName"
                                value={smtpSettings.fromName}
                                onChange={(e) => setSmtpSettings({ ...smtpSettings, fromName: e.target.value })}
                                disabled={!isEditing}
                                placeholder="Your Company Name"
                            />
                        </div>

                        {/* From Email */}
                        <div className="space-y-2">
                            <Label htmlFor="fromEmail">אימייל השולח</Label>
                            <Input
                                id="fromEmail"
                                value={smtpSettings.fromEmail}
                                onChange={(e) => setSmtpSettings({ ...smtpSettings, fromEmail: e.target.value })}
                                disabled={!isEditing}
                                placeholder="noreply@yourdomain.com"
                            />
                        </div>
                    </div>

                    {/* Secure Connection */}
                    <div className="mt-6 flex items-center space-x-2 space-x-reverse">
                        <Switch
                            id="secure"
                            checked={smtpSettings.secure}
                            onCheckedChange={(checked) => setSmtpSettings({ ...smtpSettings, secure: checked })}
                            disabled={!isEditing}
                        />
                        <Label htmlFor="secure">חיבור מאובטח (SSL/TLS)</Label>
                    </div>

                    {/* Test Connection */}
                    <div className="mt-8 pt-6 border-t border-gray-200">
                        <div className="flex justify-between items-center mb-4">
                            <h3 className="text-lg font-semibold text-gray-900">בדיקת חיבור</h3>
                            <Button
                                onClick={testConnection}
                                disabled={isTesting}
                                className="bg-blue-600 hover:bg-blue-700"
                            >
                                {isTesting ? (
                                    <>
                                        <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                                        בודק חיבור...
                                    </>
                                ) : (
                                    <>
                                        <TestTube className="w-4 h-4 ml-2" />
                                        בדוק חיבור
                                    </>
                                )}
                            </Button>
                        </div>

                        {/* Test Result */}
                        {testResult && (
                            <div className="mt-4">
                                <Card className={`p-4 ${testResult.success ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200'}`}>
                                    <div className="flex items-center">
                                        {testResult.success ? (
                                            <CheckCircle className="w-5 h-5 text-green-600 ml-3" />
                                        ) : (
                                            <XCircle className="w-5 h-5 text-red-600 ml-3" />
                                        )}
                                        <div>
                                            <h4 className={`font-semibold ${testResult.success ? 'text-green-800' : 'text-red-800'}`}>
                                                {testResult.success ? 'החיבור הצליח!' : 'החיבור נכשל'}
                                            </h4>
                                            <p className={`text-sm ${testResult.success ? 'text-green-700' : 'text-red-700'}`}>
                                                {testResult.success ? testResult.message : testResult.error}
                                            </p>
                                        </div>
                                    </div>
                                </Card>
                            </div>
                        )}

                        {/* Help Text */}
                        <div className="mt-6 text-sm text-gray-600">
                            <h4 className="font-semibold mb-2">הגדרות מומלצות:</h4>
                            <ul className="list-disc list-inside space-y-1 text-right">
                                <li>פורט 465 עם SSL/TLS מופעל (מומלץ)</li>
                                <li>פורט 587 עם STARTTLS (חלופה)</li>
                                <li>וודא שהסיסמה נכונה ושהחשבון פעיל</li>
                                <li>בדוק שהשרת מאפשר חיבורים חיצוניים</li>
                            </ul>
                        </div>

                        {/* Connection Issue Notice */}
                        <div className="mt-6 bg-yellow-50 border-l-4 border-yellow-400 p-4 text-right rounded-lg">
                            <div className="flex items-center justify-end mb-2">
                                <h4 className="text-lg font-semibold text-yellow-800 ml-2">הודעה חשובה</h4>
                                <AlertCircle className="w-5 h-5 text-yellow-600" />
                            </div>
                            <p className="text-sm text-yellow-700">
                                <strong>בעיית חיבור זוהתה:</strong> השרת mail.tsm.co.il לא זמין או חסום על ידי חומת האש.
                                <br />
                                <strong>פתרונות מומלצים:</strong>
                            </p>
                            <ul className="list-disc pr-5 text-sm text-yellow-700 space-y-1 mt-2">
                                <li>בדוק עם ספק האירוח שהפורטים 465 (SSL) ו-587 (STARTTLS) פתוחים</li>
                                <li>נסה עם שרת SMTP אחר (Gmail, Outlook, וכו')</li>
                                <li>וודא שהחשבון פעיל ומוגדר לשליחת אימיילים</li>
                                <li>בדוק את הגדרות חומת האש ברשת שלך</li>
                            </ul>
                        </div>

                        {/* Quick Test Options */}
                        <div className="mt-6 bg-gradient-to-br from-indigo-50 to-purple-50 border-2 border-indigo-300 p-6 text-right rounded-xl shadow-sm">
                            <div className="flex items-center justify-end mb-4">
                                <h4 className="text-xl font-bold text-indigo-900 ml-2">🧪 בדיקות מהירות</h4>
                                <TestTube className="w-6 h-6 text-indigo-600" />
                            </div>
                            <p className="text-sm text-indigo-700 mb-4 font-medium">
                                לחץ על הכפתורים לבדיקת קומבינציות שונות:
                            </p>

                            {/* Buttons in 2 rows of 2 */}
                            <div className="space-y-3">
                                {/* Row 1: sales@tsm.co.il */}
                                <div className="flex items-center justify-center gap-3">
                                    <Button
                                        onClick={() => quickTest('sales@tsm.co.il', '465')}
                                        disabled={isTesting}
                                        className="bg-gradient-to-r from-green-500 to-green-600 hover:from-green-600 hover:to-green-700 text-white font-semibold px-4 py-3 shadow-md hover:shadow-lg transition-all flex-1"
                                    >
                                        <TestTube className="w-4 h-4 ml-2" />
                                        sales@tsm : 465 (SSL)
                                    </Button>
                                    <Button
                                        onClick={() => quickTest('sales@tsm.co.il', '587')}
                                        disabled={isTesting}
                                        className="bg-gradient-to-r from-blue-500 to-blue-600 hover:from-blue-600 hover:to-blue-700 text-white font-semibold px-4 py-3 shadow-md hover:shadow-lg transition-all flex-1"
                                    >
                                        <TestTube className="w-4 h-4 ml-2" />
                                        sales@tsm : 587
                                    </Button>
                                </div>

                                {/* Row 2: sales (without @) */}
                                <div className="flex items-center justify-center gap-3">
                                    <Button
                                        onClick={() => quickTest('sales', '465')}
                                        disabled={isTesting}
                                        className="bg-gradient-to-r from-purple-500 to-purple-600 hover:from-purple-600 hover:to-purple-700 text-white font-semibold px-4 py-3 shadow-md hover:shadow-lg transition-all flex-1"
                                    >
                                        <TestTube className="w-4 h-4 ml-2" />
                                        sales (בלי @) : 465 (SSL)
                                    </Button>
                                    <Button
                                        onClick={() => quickTest('sales', '587')}
                                        disabled={isTesting}
                                        className="bg-gradient-to-r from-orange-500 to-orange-600 hover:from-orange-600 hover:to-orange-700 text-white font-semibold px-4 py-3 shadow-md hover:shadow-lg transition-all flex-1"
                                    >
                                        <TestTube className="w-4 h-4 ml-2" />
                                        sales (בלי @) : 587
                                    </Button>
                                </div>

                                {/* Row 3: Port 25 (No TLS) */}
                                <div className="flex items-center justify-center gap-3">
                                    <Button
                                        onClick={() => quickTest('sales@tsm.co.il', '25')}
                                        disabled={isTesting}
                                        className="bg-gradient-to-r from-red-500 to-red-600 hover:from-red-600 hover:to-red-700 text-white font-semibold px-4 py-3 shadow-md hover:shadow-lg transition-all flex-1"
                                    >
                                        <TestTube className="w-4 h-4 ml-2" />
                                        sales@tsm : 25 (No TLS)
                                    </Button>
                                    <Button
                                        onClick={() => quickTest('sales', '25')}
                                        disabled={isTesting}
                                        className="bg-gradient-to-r from-pink-500 to-pink-600 hover:from-pink-600 hover:to-pink-700 text-white font-semibold px-4 py-3 shadow-md hover:shadow-lg transition-all flex-1"
                                    >
                                        <TestTube className="w-4 h-4 ml-2" />
                                        sales (בלי @) : 25 (No TLS)
                                    </Button>
                                </div>
                            </div>
                        </div>

                        {/* SMTP Settings Info */}
                        <div className="mt-6 bg-blue-50 border-l-4 border-blue-400 p-4 text-right rounded-lg">
                            <div className="flex items-center justify-end mb-2">
                                <h4 className="text-lg font-semibold text-blue-800 ml-2">הגדרות SMTP נכונות</h4>
                                <Info className="w-5 h-5 text-blue-600" />
                            </div>
                            <div className="text-sm text-blue-700 space-y-2">
                                <p><strong>שרת:</strong> mail.tsm.co.il</p>
                                <p><strong>פורט מומלץ:</strong> 465 (SSL/TLS)</p>
                                <p><strong>פורט חלופי:</strong> 587 (STARTTLS)</p>
                                <p><strong>משתמש:</strong> sales@tsm.co.il</p>
                                <p><strong>אימות:</strong> נדרש</p>
                            </div>
                        </div>
                    </div>
                </Card>
            </div>
        </AppLayout>
    );
}
