'use client';

import { useState } from 'react';
import AppLayout from '@/src/components/layout/AppLayout';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { CheckCircle, XCircle, Clock, Search, Filter, Mail } from 'lucide-react';

export default function EmailHistoryPage() {
    const [searchQuery, setSearchQuery] = useState('');

    // נתונים דמה - לאחר מכן נחבר ל-DB
    const emailHistory = [
        {
            id: 1,
            to: 'yossi@example.com',
            toName: 'יוסי כהן',
            subject: 'תזכורת תשלום אחסון עבור example.com',
            template: 'תזכורת תשלום סטנדרטית',
            status: 'delivered',
            sentAt: '2025-10-07 14:30',
            amount: 150
        },
        {
            id: 2,
            to: 'david@example.com',
            toName: 'דוד לוי',
            subject: 'תזכורת תשלום אחסון עבור mysite.co.il',
            template: 'תזכורת תשלום סטנדרטית',
            status: 'delivered',
            sentAt: '2025-10-07 14:25',
            amount: 200
        },
        {
            id: 3,
            to: 'invalid@wrong',
            toName: 'משה ישראלי',
            subject: 'הודעה דחופה - איחור בתשלום אחסון',
            template: 'הודעת איחור תשלום',
            status: 'failed',
            sentAt: '2025-10-07 14:15',
            amount: 180,
            error: 'כתובת מייל לא תקינה'
        },
        {
            id: 4,
            to: 'sara@example.com',
            toName: 'שרה כהן',
            subject: 'אישור קבלת תשלום אחסון',
            template: 'תודה על התשלום',
            status: 'delivered',
            sentAt: '2025-10-07 13:45',
            amount: 120
        },
        {
            id: 5,
            to: 'avi@example.com',
            toName: 'אבי מזרחי',
            subject: 'תזכורת תשלום אחסון עבור shop.co.il',
            template: 'תזכורת תשלום סטנדרטית',
            status: 'pending',
            sentAt: '2025-10-07 14:35',
            amount: 300
        },
    ];

    const filteredHistory = emailHistory.filter(email =>
        email.toName.toLowerCase().includes(searchQuery.toLowerCase()) ||
        email.to.toLowerCase().includes(searchQuery.toLowerCase()) ||
        email.subject.toLowerCase().includes(searchQuery.toLowerCase())
    );

    const getStatusBadge = (status: string) => {
        switch (status) {
            case 'delivered':
                return (
                    <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
                        <CheckCircle className="w-3 h-3 ml-1" />
                        נשלח בהצלחה
                    </span>
                );
            case 'failed':
                return (
                    <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">
                        <XCircle className="w-3 h-3 ml-1" />
                        נכשל
                    </span>
                );
            case 'pending':
                return (
                    <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                        <Clock className="w-3 h-3 ml-1" />
                        ממתין
                    </span>
                );
            default:
                return null;
        }
    };

    return (
        <AppLayout>
            <div className="p-8">
                <div className="mb-8">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">היסטוריית שליחות</h1>
                    <p className="text-gray-600">כל המיילים שנשלחו מהמערכת</p>
                </div>

                {/* Search and Filters */}
                <Card className="p-6 mb-6">
                    <div className="flex items-center gap-4">
                        <div className="flex-1 relative">
                            <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                            <Input
                                type="text"
                                placeholder="חיפוש לפי שם, אימייל או נושא..."
                                value={searchQuery}
                                onChange={(e) => setSearchQuery(e.target.value)}
                                className="pr-10"
                            />
                        </div>
                        <Button variant="outline" className="gap-2">
                            <Filter className="w-4 h-4" />
                            סינון
                        </Button>
                    </div>
                </Card>

                {/* Email History Table */}
                <Card className="overflow-hidden">
                    <div className="overflow-x-auto">
                        <table className="w-full">
                            <thead className="bg-gray-50">
                                <tr>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        נמען
                                    </th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        נושא
                                    </th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        תבנית
                                    </th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        סכום
                                    </th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        תאריך
                                    </th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        סטטוס
                                    </th>
                                </tr>
                            </thead>
                            <tbody className="bg-white divide-y divide-gray-200">
                                {filteredHistory.map((email) => (
                                    <tr key={email.id} className="hover:bg-gray-50">
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="flex items-center">
                                                <div className="flex-shrink-0 h-10 w-10 bg-blue-100 rounded-full flex items-center justify-center">
                                                    <Mail className="h-5 w-5 text-blue-600" />
                                                </div>
                                                <div className="mr-4">
                                                    <div className="text-sm font-medium text-gray-900">{email.toName}</div>
                                                    <div className="text-sm text-gray-500">{email.to}</div>
                                                </div>
                                            </div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="text-sm text-gray-900">{email.subject}</div>
                                            {email.error && (
                                                <div className="text-xs text-red-600 mt-1">שגיאה: {email.error}</div>
                                            )}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="text-sm text-gray-900">{email.template}</div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="text-sm font-medium text-gray-900">₪{email.amount}</div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="text-sm text-gray-500">{email.sentAt}</div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            {getStatusBadge(email.status)}
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>

                    {filteredHistory.length === 0 && (
                        <div className="text-center py-12">
                            <Mail className="mx-auto h-12 w-12 text-gray-400" />
                            <h3 className="mt-2 text-sm font-medium text-gray-900">אין תוצאות</h3>
                            <p className="mt-1 text-sm text-gray-500">נסה לשנות את מונחי החיפוש</p>
                        </div>
                    )}
                </Card>
            </div>
        </AppLayout>
    );
}



