import React from 'react';
import { headers } from 'next/headers';
import { Customer, DashboardStats } from '@/src/types/customer';
import { DashboardGrid } from '@/components/customers/DashboardGrid';
import { CustomerTable } from '@/components/customers/CustomerTable';
import AppLayout from '@/src/components/layout/AppLayout';

// Server-side data fetching
async function getBaseUrl(): Promise<string> {
    const h = await headers();
    const host = h.get('x-forwarded-host') || h.get('host') || 'localhost:3001';
    const proto = h.get('x-forwarded-proto') || 'http';
    return `${proto}://${host}`;
}

async function getCustomers(): Promise<Customer[]> {
    const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || await getBaseUrl();
    const response = await fetch(`${baseUrl}/api/customers`, { cache: 'no-store' });
    if (!response.ok) {
        throw new Error('Failed to fetch customers');
    }
    return response.json();
}

async function getStats(): Promise<DashboardStats> {
    const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || await getBaseUrl();
    const response = await fetch(`${baseUrl}/api/dashboard/stats`, { cache: 'no-store' });
    if (!response.ok) {
        throw new Error('Failed to fetch stats');
    }
    return response.json();
}

export default async function CustomersPage() {
    let customers: Customer[] = [];
    let stats: DashboardStats | null = null;
    let error: string | null = null;

    try {
        [customers, stats] = await Promise.all([getCustomers(), getStats()]);
    } catch (err) {
        error = err instanceof Error ? err.message : 'An error occurred';
    }

    if (error) {
        return (
            <AppLayout>
                <div className="flex items-center justify-center h-full">
                    <div className="text-center p-6 bg-white rounded-lg shadow-md">
                        <h2 className="text-xl font-semibold text-red-600">שגיאה בטעינת נתונים</h2>
                        <p className="text-gray-700 mt-2">{error}</p>
                    </div>
                </div>
            </AppLayout>
        );
    }

    return (
        <AppLayout>
            <div className="min-h-screen bg-gray-50">
                <div className="max-w-7xl mx-auto px-4 py-8">
                    {/* Modern Header */}
                    <div className="mb-10">
                        <div className="flex items-center justify-between">
                            <div>
                                <h1 className="text-4xl font-bold text-gray-900 mb-2">ניהול לקוחות</h1>
                                <p className="text-lg text-gray-600">דשבורד וניהול לקוחות מתקדם</p>
                            </div>
                            <a
                                href="/customers/add"
                                className="px-6 py-3 bg-white border-2 border-blue-600 text-blue-600 rounded-xl hover:bg-blue-50 hover:border-blue-700 transition-all duration-300 shadow-lg hover:shadow-xl font-semibold"
                            >
                                + הוסף לקוח חדש
                            </a>
                        </div>
                    </div>

                    {/* Modern Dashboard */}
                    {stats && (
                        <div className="mb-12">
                            <div className="mb-6">
                                <h2 className="text-2xl font-bold text-gray-900 mb-2">סטטיסטיקות כלליות</h2>
                                <p className="text-gray-600">סקירה מהירה של הנתונים החשובים</p>
                            </div>
                            <DashboardGrid stats={stats} />
                        </div>
                    )}

                    {/* Customers Section */}
                    <div className="bg-white rounded-2xl shadow-xl border border-gray-100 overflow-hidden">
                        <div className="px-8 py-6 border-b border-gray-100 bg-gradient-to-r from-gray-50 to-white">
                            <div className="flex items-center justify-between">
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900">לקוחות</h2>
                                    <p className="text-gray-600 mt-1">{customers.length} לקוחות רשומים במערכת</p>
                                </div>
                                <div className="flex items-center gap-3">
                                    <div className="px-4 py-2 bg-blue-100 text-blue-700 rounded-lg font-medium">
                                        {customers.length} סה"כ
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="p-8">
                            {customers.length > 0 ? (
                                <CustomerTable
                                    customers={customers}
                                />
                            ) : (
                                <div className="text-center py-16">
                                    <div className="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                        <svg className="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z" />
                                        </svg>
                                    </div>
                                    <h3 className="text-xl font-semibold text-gray-900 mb-2">אין לקוחות עדיין</h3>
                                    <p className="text-gray-500 mb-6">הוסף לקוח ראשון כדי להתחיל לנהל את העסק שלך</p>
                                    <a
                                        href="/customers/add"
                                        className="inline-flex items-center px-6 py-3 bg-white border-2 border-blue-600 text-blue-600 rounded-lg hover:bg-blue-50 hover:border-blue-700 transition-colors font-medium"
                                    >
                                        + הוסף לקוח ראשון
                                    </a>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
