import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { ComprehensiveStats } from '@/src/types/comprehensive';
import Sidebar from '@/src/components/layout/Sidebar';
import Header from '@/src/components/layout/Header';

async function getComprehensiveStats(): Promise<ComprehensiveStats> {
    await initComprehensiveDatabase();
    const db = getComprehensiveDatabase();

    // Customer statistics
    const customerStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN status = 'lead' THEN 1 ELSE 0 END) as leads,
            SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive
        FROM customers
    `).get() as unknown;

    // Domain statistics
    const domainStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN status = 'expired' THEN 1 ELSE 0 END) as expired,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending
        FROM domains
    `).get() as unknown;

    // Project statistics
    const projectStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
            SUM(CASE WHEN status = 'planning' THEN 1 ELSE 0 END) as planning
        FROM projects
    `).get() as unknown;

    // Quote statistics
    const quoteStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
            SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
        FROM quotes
    `).get() as unknown;

    // Payment statistics
    const paymentStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
            SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
            COALESCE(SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END), 0) as revenue
        FROM payments
    `).get() as unknown;

    // Activity statistics
    const activityStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
            SUM(CASE WHEN due_date < datetime('now') AND status = 'pending' THEN 1 ELSE 0 END) as overdue
        FROM activities
    `).get() as unknown;

    // Reminder statistics
    const reminderStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN reminder_date < datetime('now') AND status = 'pending' THEN 1 ELSE 0 END) as overdue,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed
        FROM reminders
    `).get() as unknown;

    // Service statistics
    const emailServiceStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active
        FROM email_services
    `).get() as unknown;

    const technicalServiceStats = db.prepare(`
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active
        FROM technical_services
    `).get() as unknown;

    // Calculate overdue projects (projects past end_date)
    const overdueProjects = db.prepare(`
        SELECT COUNT(*) as count
        FROM projects 
        WHERE end_date < date('now') AND status IN ('open', 'in_progress', 'pending')
    `).get() as unknown;

    // Calculate overdue payments (payments past due_date)
    const overduePayments = db.prepare(`
        SELECT COUNT(*) as count
        FROM payments 
        WHERE due_date < datetime('now') AND status = 'pending'
    `).get() as unknown;

    // Calculate expiring domains (expiry within 30 days)
    const expiringDomains = db.prepare(`
        SELECT COUNT(*) as count
        FROM domains 
        WHERE date(expiry_date) <= date('now', '+30 days') AND status = 'active'
    `).get() as unknown;

    const stats: ComprehensiveStats = {
        // Customer stats
        totalCustomers: customerStats.total || 0,
        activeCustomers: customerStats.active || 0,
        leadCustomers: customerStats.leads || 0,
        inactiveCustomers: customerStats.inactive || 0,

        // Domain stats
        totalDomains: domainStats.total || 0,
        activeDomains: domainStats.active || 0,
        expiringDomains: expiringDomains.count || 0,
        expiredDomains: domainStats.expired || 0,

        // Project stats
        totalProjects: projectStats.total || 0,
        activeProjects: projectStats.active || 0,
        completedProjects: projectStats.completed || 0,
        overdueProjects: overdueProjects.count || 0,

        // Quote stats
        totalQuotes: quoteStats.total || 0,
        pendingQuotes: quoteStats.pending || 0,
        approvedQuotes: quoteStats.approved || 0,
        rejectedQuotes: quoteStats.rejected || 0,

        // Payment stats
        totalPayments: paymentStats.total || 0,
        pendingPayments: paymentStats.pending || 0,
        completedPayments: paymentStats.completed || 0,
        overduePayments: overduePayments.count || 0,
        totalRevenue: paymentStats.revenue || 0,

        // Activity stats
        totalActivities: activityStats.total || 0,
        pendingActivities: activityStats.pending || 0,
        completedActivities: activityStats.completed || 0,
        overdueActivities: activityStats.overdue || 0,

        // Reminder stats
        totalReminders: reminderStats.total || 0,
        pendingReminders: reminderStats.pending || 0,
        overdueReminders: reminderStats.overdue || 0,
        completedReminders: reminderStats.completed || 0,

        // Service stats
        totalEmailServices: emailServiceStats.total || 0,
        activeEmailServices: emailServiceStats.active || 0,
        totalTechnicalServices: technicalServiceStats.total || 0,
        activeTechnicalServices: technicalServiceStats.active || 0,
    };

    return stats;
}

export default async function ComprehensiveDashboard() {
    const stats = await getComprehensiveStats();

    return (
        <div className="min-h-screen bg-gray-50 flex">
            <Sidebar />
            <div className="flex-1 flex flex-col">
                <Header />
                <div className="flex-1 p-6">
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900 mb-2">
                            🎯 דשבורד מקיף - מערכת CRM מתקדמת
                        </h1>
                        <p className="text-gray-600">
                            ניהול מקיף של לקוחות, דומיינים, פרויקטים, תשלומים ופעילויות
                        </p>
                    </div>

                    {/* Overview Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                        <div className="bg-white p-6 rounded-lg shadow-md border-l-4 border-blue-500">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-gray-600">סה"כ לקוחות</p>
                                    <p className="text-2xl font-bold text-gray-900">{stats.totalCustomers}</p>
                                </div>
                                <div className="text-blue-500">
                                    <svg className="w-8 h-8" fill="currentColor" viewBox="0 0 20 20">
                                        <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                    </svg>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white p-6 rounded-lg shadow-md border-l-4 border-green-500">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-gray-600">סה"כ דומיינים</p>
                                    <p className="text-2xl font-bold text-gray-900">{stats.totalDomains}</p>
                                </div>
                                <div className="text-green-500">
                                    <svg className="w-8 h-8" fill="currentColor" viewBox="0 0 20 20">
                                        <path fillRule="evenodd" d="M12.586 4.586a2 2 0 112.828 2.828l-3 3a2 2 0 01-2.828 0 1 1 0 00-1.414 1.414 4 4 0 005.656 0l3-3a4 4 0 00-5.656-5.656l-1.5 1.5a1 1 0 101.414 1.414l1.5-1.5zm-5 5a2 2 0 012.828 0 1 1 0 101.414-1.414 4 4 0 00-5.656 0l-3 3a4 4 0 105.656 5.656l1.5-1.5a1 1 0 10-1.414-1.414l-1.5 1.5a2 2 0 11-2.828-2.828l3-3z" clipRule="evenodd" />
                                    </svg>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white p-6 rounded-lg shadow-md border-l-4 border-purple-500">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-gray-600">סה"כ פרויקטים</p>
                                    <p className="text-2xl font-bold text-gray-900">{stats.totalProjects}</p>
                                </div>
                                <div className="text-purple-500">
                                    <svg className="w-8 h-8" fill="currentColor" viewBox="0 0 20 20">
                                        <path fillRule="evenodd" d="M6 2a1 1 0 00-1 1v1H4a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2h-1V3a1 1 0 10-2 0v1H7V3a1 1 0 00-1-1zm0 5a1 1 0 000 2h8a1 1 0 100-2H6z" clipRule="evenodd" />
                                    </svg>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white p-6 rounded-lg shadow-md border-l-4 border-yellow-500">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-gray-600">סה"כ הכנסות</p>
                                    <p className="text-2xl font-bold text-gray-900">₪{stats.totalRevenue.toLocaleString()}</p>
                                </div>
                                <div className="text-yellow-500">
                                    <svg className="w-8 h-8" fill="currentColor" viewBox="0 0 20 20">
                                        <path d="M8.433 7.418c.155-.103.346-.196.567-.267v1.698a2.305 2.305 0 01-.567-.267C8.07 8.34 8 8.114 8 8c0-.114.07-.34.433-.582zM11 12.849v-1.698c.22.071.412.164.567.267.364.243.433.468.433.582 0 .114-.07.34-.433.582a2.305 2.305 0 01-.567.267z" />
                                        <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-13a1 1 0 10-2 0v.092a4.535 4.535 0 00-1.676.662C6.602 6.234 6 7.009 6 8c0 .99.602 1.765 1.324 2.246.48.32 1.054.545 1.676.662v1.941c-.391-.127-.68-.317-.843-.504a1 1 0 10-1.51 1.31c.562.649 1.413 1.076 2.353 1.253V15a1 1 0 102 0v-.092a4.535 4.535 0 001.676-.662C13.398 13.766 14 12.991 14 12c0-.99-.602-1.765-1.324-2.246A4.535 4.535 0 0011 9.092V7.151c.391.127.68.317.843.504a1 1 0 101.511-1.31c-.563-.649-1.413-1.076-2.354-1.253V5z" clipRule="evenodd" />
                                    </svg>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Detailed Statistics */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                        {/* Left Column - Customers & Domains */}
                        <div className="space-y-6">
                            <div className="bg-white p-6 rounded-lg shadow-md">
                                <h3 className="text-lg font-semibold text-gray-900 mb-4">👥 סטטיסטיקות לקוחות</h3>
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-green-600">{stats.activeCustomers}</p>
                                        <p className="text-sm text-gray-600">לקוחות פעילים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-blue-600">{stats.leadCustomers}</p>
                                        <p className="text-sm text-gray-600">לידים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-gray-600">{stats.inactiveCustomers}</p>
                                        <p className="text-sm text-gray-600">לא פעילים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-purple-600">{stats.totalCustomers}</p>
                                        <p className="text-sm text-gray-600">סה"כ</p>
                                    </div>
                                </div>
                            </div>

                            <div className="bg-white p-6 rounded-lg shadow-md">
                                <h3 className="text-lg font-semibold text-gray-900 mb-4">🌐 סטטיסטיקות דומיינים</h3>
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-green-600">{stats.activeDomains}</p>
                                        <p className="text-sm text-gray-600">דומיינים פעילים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-yellow-600">{stats.expiringDomains}</p>
                                        <p className="text-sm text-gray-600">דומיינים פגים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-red-600">{stats.expiredDomains}</p>
                                        <p className="text-sm text-gray-600">דומיינים פגו</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-purple-600">{stats.totalDomains}</p>
                                        <p className="text-sm text-gray-600">סה"כ</p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Right Column - Projects & Payments */}
                        <div className="space-y-6">
                            <div className="bg-white p-6 rounded-lg shadow-md">
                                <h3 className="text-lg font-semibold text-gray-900 mb-4">📋 סטטיסטיקות פרויקטים</h3>
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-blue-600">{stats.activeProjects}</p>
                                        <p className="text-sm text-gray-600">פרויקטים פעילים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-green-600">{stats.completedProjects}</p>
                                        <p className="text-sm text-gray-600">הושלמו</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-red-600">{stats.overdueProjects}</p>
                                        <p className="text-sm text-gray-600">איחורים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-purple-600">{stats.totalProjects}</p>
                                        <p className="text-sm text-gray-600">סה"כ</p>
                                    </div>
                                </div>
                            </div>

                            <div className="bg-white p-6 rounded-lg shadow-md">
                                <h3 className="text-lg font-semibold text-gray-900 mb-4">💳 סטטיסטיקות תשלומים</h3>
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-green-600">{stats.completedPayments}</p>
                                        <p className="text-sm text-gray-600">תשלומים הושלמו</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-yellow-600">{stats.pendingPayments}</p>
                                        <p className="text-sm text-gray-600">ממתינים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-red-600">{stats.overduePayments}</p>
                                        <p className="text-sm text-gray-600">איחורים</p>
                                    </div>
                                    <div className="text-center">
                                        <p className="text-2xl font-bold text-purple-600">{stats.totalPayments}</p>
                                        <p className="text-sm text-gray-600">סה"כ</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Activities & Reminders */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mt-8">
                        <div className="bg-white p-6 rounded-lg shadow-md">
                            <h3 className="text-lg font-semibold text-gray-900 mb-4">📝 פעילויות</h3>
                            <div className="grid grid-cols-2 gap-4">
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-blue-600">{stats.pendingActivities}</p>
                                    <p className="text-sm text-gray-600">ממתינות</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-green-600">{stats.completedActivities}</p>
                                    <p className="text-sm text-gray-600">הושלמו</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-red-600">{stats.overdueActivities}</p>
                                    <p className="text-sm text-gray-600">איחורים</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-purple-600">{stats.totalActivities}</p>
                                    <p className="text-sm text-gray-600">סה"כ</p>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white p-6 rounded-lg shadow-md">
                            <h3 className="text-lg font-semibold text-gray-900 mb-4">⏰ תזכורות</h3>
                            <div className="grid grid-cols-2 gap-4">
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-yellow-600">{stats.pendingReminders}</p>
                                    <p className="text-sm text-gray-600">ממתינות</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-green-600">{stats.completedReminders}</p>
                                    <p className="text-sm text-gray-600">הושלמו</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-red-600">{stats.overdueReminders}</p>
                                    <p className="text-sm text-gray-600">איחורים</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-purple-600">{stats.totalReminders}</p>
                                    <p className="text-sm text-gray-600">סה"כ</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Services */}
                    <div className="mt-8">
                        <div className="bg-white p-6 rounded-lg shadow-md">
                            <h3 className="text-lg font-semibold text-gray-900 mb-4">⚙️ שירותים טכניים</h3>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-green-600">{stats.activeEmailServices}</p>
                                    <p className="text-sm text-gray-600">שירותי אימייל פעילים</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-purple-600">{stats.totalEmailServices}</p>
                                    <p className="text-sm text-gray-600">סה"כ שירותי אימייל</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-green-600">{stats.activeTechnicalServices}</p>
                                    <p className="text-sm text-gray-600">שירותים טכניים פעילים</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-2xl font-bold text-purple-600">{stats.totalTechnicalServices}</p>
                                    <p className="text-sm text-gray-600">סה"כ שירותים טכניים</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}
