import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

export async function GET(request: NextRequest) {
    const db = getDb();
    try {
        // סטטיסטיקות בסיסיות
        const totalStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'on_hold' THEN 1 ELSE 0 END) as on_hold,
                SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled
            FROM tasks
        `).get() as unknown as {
            total: number;
            pending: number;
            in_progress: number;
            completed: number;
            on_hold: number;
            cancelled: number;
        };

        // סטטיסטיקות עדיפות
        const priorityStats = db.prepare(`
            SELECT 
                SUM(CASE WHEN priority = 'urgent' THEN 1 ELSE 0 END) as urgent,
                SUM(CASE WHEN priority = 'high' THEN 1 ELSE 0 END) as high,
                SUM(CASE WHEN priority = 'medium' THEN 1 ELSE 0 END) as medium,
                SUM(CASE WHEN priority = 'low' THEN 1 ELSE 0 END) as low
            FROM tasks
        `).get() as unknown as {
            urgent: number;
            high: number;
            medium: number;
            low: number;
        };

        // משימות באיחור (due_date < היום)
        const overdueStats = db.prepare(`
            SELECT COUNT(*) as overdue
            FROM tasks 
            WHERE due_date < date('now') 
            AND status NOT IN ('completed', 'cancelled')
        `).get() as unknown as { overdue: number };

        // משימות היום (due_date = היום)
        const todayStats = db.prepare(`
            SELECT COUNT(*) as due_today
            FROM tasks 
            WHERE due_date = date('now') 
            AND status NOT IN ('completed', 'cancelled')
        `).get() as unknown as { due_today: number };

        // משימות השבוע (due_date בין היום ל-7 ימים)
        const weekStats = db.prepare(`
            SELECT COUNT(*) as due_this_week
            FROM tasks 
            WHERE due_date BETWEEN date('now') AND date('now', '+7 days')
            AND status NOT IN ('completed', 'cancelled')
        `).get() as unknown as { due_this_week: number };

        const stats = {
            ...totalStats,
            ...priorityStats,
            ...overdueStats,
            ...todayStats,
            ...weekStats,
        };

        return NextResponse.json(stats);
    } catch (e: any) {
        console.error('Get tasks stats error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת סטטיסטיקות: ' + e.message }, { status: 500 });
    } finally {
        db.close();
    }
}
