import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

export async function GET(request: NextRequest) {
    const db = getDb();
    try {
        const url = new URL(request.url);
        const status = url.searchParams.get('status');
        const priority = url.searchParams.get('priority');
        const customerId = url.searchParams.get('customer_id');
        const domainId = url.searchParams.get('domain_id');
        const projectId = url.searchParams.get('project_id');
        const eventId = url.searchParams.get('event_id');
        const hasEvent = url.searchParams.get('has_event');
        const q = url.searchParams.get('q');

        const where: string[] = [];
        const params: Record<string, unknown> = {};
        if (status) { where.push('t.status = @status'); params.status = status; }
        if (priority) { where.push('t.priority = @priority'); params.priority = priority; }
        if (customerId) { where.push('t.customer_id = @customerId'); params.customerId = customerId; }
        if (domainId) { where.push('t.domain_id = @domainId'); params.domainId = domainId; }
        if (projectId) { where.push('t.project_id = @projectId'); params.projectId = projectId; }
        if (eventId) { where.push('t.event_id = @eventId'); params.eventId = eventId; }
        if (hasEvent === 'true') { where.push('t.event_id IS NOT NULL'); }
        if (hasEvent === 'false') { where.push('t.event_id IS NULL'); }
        if (q) { where.push('(t.title LIKE @q OR t.description LIKE @q)'); params.q = `%${q}%`; }

        const whereSql = where.length ? `WHERE ${where.join(' AND ')}` : '';
        const sql = `SELECT 
            t.*,
            c.name as customer_name,
            p.name as project_name,
            tc.name as category_name
        FROM tasks t 
        LEFT JOIN customers c ON t.customer_id = c.id
        LEFT JOIN projects p ON t.project_id = p.id
        LEFT JOIN tasks_categories tc ON t.category_id = tc.id
        ${whereSql} 
        ORDER BY 
            CASE t.priority WHEN 'urgent' THEN 1 WHEN 'high' THEN 2 WHEN 'medium' THEN 3 ELSE 4 END,
            COALESCE(t.due_date, '9999-12-31') ASC`;
        const rows = db.prepare(sql).all(params);
        return NextResponse.json(rows);
    } catch (e: any) {
        console.error('List tasks error:', e);
        return NextResponse.json({ error: e.message }, { status: 500 });
    } finally {
        db.close();
    }
}

export async function POST(request: NextRequest) {
    const db = getDb();
    try {
        console.log('🔄 Creating new task...');
        const body = await request.json();
        console.log('📝 Request body:', body);

        // וולידציה: customer_id חובה
        if (!body.customer_id) {
            console.log('❌ Missing customer_id');
            return NextResponse.json(
                { error: 'שדה לקוח הוא חובה' },
                { status: 400 }
            );
        }

        // וולידציה: אם יש domain_id, לוודא שהוא שייך ללקוח
        if (body.domain_id) {
            const domain = db.prepare('SELECT customer_id FROM domains WHERE id = ?').get(body.domain_id) as unknown;
            if (!domain) {
                return NextResponse.json(
                    { error: 'דומיין לא נמצא' },
                    { status: 400 }
                );
            }
            if (domain.customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הדומיין לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        // וולידציה: אם יש project_id, לוודא שהוא שייך ללקוח
        if (body.project_id) {
            const project = db.prepare('SELECT customer_id FROM projects WHERE id = ?').get(body.project_id) as unknown;
            if (!project) {
                return NextResponse.json(
                    { error: 'פרויקט לא נמצא' },
                    { status: 400 }
                );
            }
            if (project.customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הפרויקט לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        // יצירת ID ייחודי למשימה
        const taskId = `task_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

        // הכנת נתוני המשימה בהתאם למבנה הטבלה הקיים
        const taskData = {
            id: taskId,
            title: body.title,
            description: body.description || null,
            status: body.status || 'pending',
            priority: body.priority || 'medium',
            category_id: body.category_id || null,
            customer_id: body.customer_id,
            project_id: body.project_id || null,
            assigned_to: body.assigned_to || null,
            due_date: body.due_date || null,
            completed_at: body.completed_at || null
        };

        console.log('📊 Task data to insert:', taskData);

        const insert = db.prepare(`INSERT INTO tasks (
            id, title, description, status, priority, category_id, customer_id, project_id, 
            assigned_to, due_date, completed_at
        ) VALUES (
            @id, @title, @description, @status, @priority, @category_id, @customer_id, @project_id,
            @assigned_to, @due_date, @completed_at
        )`);
        const info = insert.run(taskData);
        console.log('✅ Task inserted with ID:', taskId);

        const task = db.prepare('SELECT * FROM tasks WHERE id = ?').get(taskId);
        console.log('📋 Created task:', task);

        return NextResponse.json(task, { status: 201 });
    } catch (e: any) {
        console.error('Create task error:', e);
        return NextResponse.json({
            error: e.message.includes('Domain does not belong')
                ? 'הדומיין לא שייך ללקוח שנבחר'
                : 'שגיאה ביצירת משימה: ' + e.message
        }, { status: 500 });
    } finally {
        db.close();
    }
}


