import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

export async function POST(request: NextRequest) {
    const db = getDb();
    try {
        console.log('🔧 Creating tasks table in comprehensive.db...');

        // Create tasks table
        db.exec(`
            CREATE TABLE IF NOT EXISTS tasks (
                id TEXT PRIMARY KEY,
                title TEXT NOT NULL,
                description TEXT,
                status TEXT DEFAULT 'pending',
                priority TEXT DEFAULT 'medium',
                category_id TEXT,
                customer_id TEXT,
                project_id TEXT,
                assigned_to TEXT,
                due_date TEXT,
                completed_at TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES customers(id),
                FOREIGN KEY (project_id) REFERENCES projects(id)
            )
        `);
        console.log('✅ Tasks table created');

        // Create tasks_categories table
        db.exec(`
            CREATE TABLE IF NOT EXISTS tasks_categories (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                color TEXT DEFAULT '#3B82F6',
                icon TEXT DEFAULT '📋',
                created_at TEXT DEFAULT CURRENT_TIMESTAMP
            )
        `);
        console.log('✅ Tasks categories table created');

        // Insert default categories
        const categories = [
            { id: 'cat_1', name: 'כללי', color: '#3B82F6', icon: '📋' },
            { id: 'cat_2', name: 'דחוף', color: '#EF4444', icon: '🔥' },
            { id: 'cat_3', name: 'פגישה', color: '#10B981', icon: '🤝' },
            { id: 'cat_4', name: 'תמיכה', color: '#F59E0B', icon: '🛠️' },
            { id: 'cat_5', name: 'פיתוח', color: '#8B5CF6', icon: '💻' }
        ];

        const insertCategory = db.prepare(`
            INSERT OR IGNORE INTO tasks_categories (id, name, color, icon, created_at)
            VALUES (?, ?, ?, ?, datetime('now'))
        `);

        categories.forEach(cat => {
            insertCategory.run(cat.id, cat.name, cat.color, cat.icon);
        });
        console.log('✅ Default categories inserted');

        // Create indexes for performance
        db.exec(`
            CREATE INDEX IF NOT EXISTS idx_tasks_status ON tasks(status);
            CREATE INDEX IF NOT EXISTS idx_tasks_priority ON tasks(priority);
            CREATE INDEX IF NOT EXISTS idx_tasks_customer ON tasks(customer_id);
            CREATE INDEX IF NOT EXISTS idx_tasks_project ON tasks(project_id);
            CREATE INDEX IF NOT EXISTS idx_tasks_due_date ON tasks(due_date);
        `);
        console.log('✅ Indexes created');

        return NextResponse.json({
            success: true,
            message: 'טבלת המשימות נוצרה בהצלחה!'
        });

    } catch (error: any) {
        console.error('❌ Error initializing tasks:', error);
        return NextResponse.json({
            error: 'שגיאה באתחול טבלת המשימות: ' + error.message
        }, { status: 500 });
    } finally {
        db.close();
    }
}
