import { NextRequest, NextResponse } from 'next/server';
import { openDB } from '@/src/lib/db';

// GET /api/projects - רשימת פרויקטים
export async function GET(request: NextRequest) {
    try {
        const db = await openDB();
        const { searchParams } = new URL(request.url);

        const page = parseInt(searchParams.get('page') || '1');
        const limit = parseInt(searchParams.get('limit') || '50');
        const status = searchParams.get('status');
        const customer_id = searchParams.get('customer_id');
        const project_type = searchParams.get('project_type');
        const search = searchParams.get('search');
        const is_open = searchParams.get('is_open');

        const offset = (page - 1) * limit;

        // בניית שאילתה דינמית
        let whereConditions = [];
        let params: any[] = [];

        if (status) {
            whereConditions.push('p.status = ?');
            params.push(status);
        }

        if (customer_id) {
            whereConditions.push('p.customer_id = ?');
            params.push(customer_id);
        }

        if (project_type) {
            whereConditions.push('p.project_type = ?');
            params.push(project_type);
        }

        if (search) {
            whereConditions.push('(p.title LIKE ? OR p.description LIKE ?)');
            params.push(`%${search}%`, `%${search}%`);
        }

        // Note: is_open field doesn't exist in projects table
        // if (is_open !== null) {
        //     whereConditions.push('p.is_open = ?');
        //     params.push(is_open === 'true' ? 1 : 0);
        // }

        const whereClause = whereConditions.length > 0 ? `WHERE ${whereConditions.join(' AND ')}` : '';

        // שאילתה פשוטה מאוד
        const query = `
      SELECT 
        p.*
      FROM projects p
      ORDER BY p.created_date DESC
      LIMIT ? OFFSET ?
    `;

        const countQuery = `
      SELECT COUNT(*) as total
      FROM projects p
    `;

        const projects = await new Promise((resolve, reject) => {
            const queryParams = [limit, offset];
            console.log('Query:', query);
            console.log('Params:', queryParams);
            db.all(query, queryParams, (err, rows) => {
                if (err) {
                    console.error('Database error:', err);
                    reject(err);
                } else {
                    console.log('Projects found:', rows?.length || 0);
                    resolve(rows);
                }
            });
        });

        const totalResult = await new Promise((resolve, reject) => {
            db.get(countQuery, [], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });

        const total = (totalResult as any)?.total || 0;

        // המרת JSON fields - גרסה פשוטה יותר
        const formattedProjects = (projects as any[]).map(project => {
            // assigned_team - אם זה string, נשמור כמערך
            let assigned_team = [];
            if (project.assigned_team) {
                if (typeof project.assigned_team === 'string') {
                    // אם זה JSON תקין, נפרסר אותו
                    if (project.assigned_team.startsWith('[') || project.assigned_team.startsWith('{')) {
                        try {
                            assigned_team = JSON.parse(project.assigned_team);
                        } catch {
                            // אם זה לא JSON, נשמור כטקסט רגיל במערך
                            assigned_team = [project.assigned_team];
                        }
                    } else {
                        // אם זה טקסט רגיל, נשמור כמערך
                        assigned_team = [project.assigned_team];
                    }
                } else if (Array.isArray(project.assigned_team)) {
                    assigned_team = project.assigned_team;
                }
            }

            return {
                ...project,
                assigned_team,
                budget: project.budget || 0,
                progress: project.progress || 0
            };
        });

        return NextResponse.json({
            projects: formattedProjects,
            pagination: {
                page,
                limit,
                total,
                pages: Math.ceil(total / limit)
            }
        });

    } catch (error) {
        console.error('Error fetching projects:', error);
        return NextResponse.json(
            { error: 'שגיאה בטעינת הפרויקטים' },
            { status: 500 }
        );
    }
}

// POST /api/projects - יצירת פרויקט חדש
export async function POST(request: NextRequest) {
    try {
        const db = await openDB();
        const body = await request.json();

        const {
            title,
            description,
            customer_id,
            status = 'planning',
            start_date,
            end_date,
            budget = 0,
            progress = 0,
            assigned_team = [],
            notes = ''
        } = body;

        // ולידציה
        if (!title || !customer_id) {
            return NextResponse.json(
                { error: 'כותרת ולקוח הם שדות חובה' },
                { status: 400 }
            );
        }

        // יצירת ID ייחודי
        const id = `project_${Date.now()}`;

        const insertQuery = `
      INSERT INTO projects (
        id, title, description, customer_id, status, 
        start_date, end_date, budget, progress, assigned_team, notes
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

        const result = await new Promise((resolve, reject) => {
            db.run(insertQuery, [
                id,
                title,
                description,
                customer_id,
                status,
                start_date,
                end_date,
                budget,
                progress,
                JSON.stringify(assigned_team),
                notes
            ], function (err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({ lastInsertRowid: this.lastInsertRowid });
                }
            });
        });

        return NextResponse.json({
            id: id,
            message: 'פרויקט נוצר בהצלחה'
        });

    } catch (error) {
        console.error('Error creating project:', error);
        return NextResponse.json(
            { error: 'שגיאה ביצירת הפרויקט' },
            { status: 500 }
        );
    }
}
