import { NextRequest, NextResponse } from 'next/server';
import { getDatabase, initDatabase } from '@/src/lib/database';
import { openDB } from '@/src/lib/db';
import { Phone } from '@/src/types/phone';

// GET - קבלת כל הפלאפונים
export async function GET() {
    try {
        await initDatabase();
        const db = await getDatabase();

        return new Promise<NextResponse>((resolve, reject) => {
            db.all('SELECT * FROM phones ORDER BY createdAt DESC', (err, rows) => {
                if (err) {
                    console.error('Error fetching phones:', err);
                    reject(NextResponse.json({ error: err.message }, { status: 500 }));
                } else {
                    resolve(NextResponse.json(rows));
                }
            });
        });
    } catch (error) {
        console.error('Database error:', error);
        return NextResponse.json({ error: 'Database error' }, { status: 500 });
    }
}

// POST - הוספת פלאפון חדש
export async function POST(request: NextRequest) {
    try {
        await initDatabase();
        const phonesDb = await getDatabase();
        const comprehensiveDb = await openDB();
        const phone: Omit<Phone, 'id'> = await request.json();

        const id = Date.now().toString();
        const now = new Date().toISOString();

        // Step 1: Check if customer exists in comprehensive.db by email
        const customerExists = await new Promise<any>((resolve) => {
            comprehensiveDb.get(
                'SELECT * FROM customers WHERE email = ?',
                [phone.email],
                (err, row) => {
                    if (err) {
                        console.error('Error checking customer:', err);
                        resolve(null);
                    } else {
                        resolve(row);
                    }
                }
            );
        });

        let customerId = customerExists?.id;

        // Step 2: If customer doesn't exist, create them
        if (!customerExists) {
            console.log(`📝 Creating new customer: ${phone.owner} (${phone.email})`);

            customerId = `customer_${Date.now()}`;

            await new Promise<void>((resolve, reject) => {
                comprehensiveDb.run(
                    `INSERT INTO customers (id, name, email, phone, company, status, stage, value, created_date, last_activity, updated_at)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                    [
                        customerId,
                        phone.owner,
                        phone.email,
                        phone.phone,
                        '', // company - empty for now
                        'active', // status
                        'quote', // stage
                        0, // value
                        now,
                        now,
                        now
                    ],
                    (err) => {
                        if (err) {
                            console.error('Error creating customer:', err);
                            reject(err);
                        } else {
                            console.log(`✅ Customer created: ${customerId} - ${phone.owner}`);
                            resolve();
                        }
                    }
                );
            });
        } else {
            console.log(`✅ Customer exists: ${customerExists.id} - ${customerExists.name}`);
        }

        // Step 3: Add domain/phone to phones.db
        return new Promise<NextResponse>((resolve, reject) => {
            const sql = `INSERT INTO phones (id, phone, domain, website, owner, email, package, amount, status, priority, startDate, nextPayment, notes, createdAt, updatedAt) 
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, COALESCE(?, date('now')), ?, ?, ?, ?)`;

            phonesDb.run(sql, [
                id, phone.phone, phone.domain, phone.website, phone.owner, phone.email,
                phone.package, phone.amount, phone.status, phone.priority,
                phone.startDate, phone.nextPayment, phone.notes || '', now, now
            ], function (err) {
                if (err) {
                    console.error('Error adding phone:', err);
                    reject(NextResponse.json({ error: err.message }, { status: 500 }));
                } else {
                    const newPhone: Phone = {
                        id,
                        ...phone,
                        createdAt: now,
                        updatedAt: now
                    };
                    console.log(`✅ Domain added successfully: ${phone.domain} → Customer: ${customerId}`);
                    resolve(NextResponse.json(newPhone));
                }
            });
        });
    } catch (error) {
        console.error('Database error:', error);
        return NextResponse.json({ error: 'Database error' }, { status: 500 });
    }
}
