/**
 * API Route: Notes
 * GET /api/notes - קבלת פתקים עם פילטרים
 * POST /api/notes - יצירת פתק חדש
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';
import { Note } from '@/types/notes';
import { generateId } from '@/src/lib/notes-helpers';

export async function GET(request: NextRequest) {
    try {
        const { searchParams } = new URL(request.url);
        const listId = searchParams.get('listId');
        const filter = searchParams.get('filter');
        const search = searchParams.get('search');

        const db = await getNotesDatabase();

        let query = 'SELECT * FROM notes';
        const conditions: string[] = [];
        const params: any[] = [];

        // Filter by list
        if (listId && listId !== 'all') {
            if (listId === 'null' || listId === 'inbox') {
                conditions.push('list_id IS NULL');
            } else {
                conditions.push('list_id = ?');
                params.push(listId);
            }
        }

        // Filter by status
        if (filter === 'open') {
            conditions.push('done = 0');
        } else if (filter === 'completed') {
            conditions.push('done = 1');
        }

        // Search
        if (search) {
            conditions.push('(title LIKE ? OR description LIKE ?)');
            params.push(`%${search}%`, `%${search}%`);
        }

        if (conditions.length > 0) {
            query += ' WHERE ' + conditions.join(' AND ');
        }

        query += ' ORDER BY created_at DESC';

        const rows = await db.all(query, params);

        const notes: Note[] = rows.map((row: any) => ({
            id: row.id,
            listId: row.list_id,
            title: row.title,
            description: row.description,
            done: row.done === 1,
            important: row.important === 1,
            flagged: row.flagged === 1,
            dueDate: row.due_date,
            reminderDate: row.reminder_date,
            reminderTime: row.reminder_time,
            reminderRecurrence: row.reminder_recurrence || 'none',
            createdAt: row.created_at,
            updatedAt: row.updated_at,
            subtasks: row.subtasks ? JSON.parse(row.subtasks) : [],
            tags: row.tags ? JSON.parse(row.tags) : [],
            attachments: row.attachments ? JSON.parse(row.attachments) : [],
        }));

        return NextResponse.json({ notes, total: notes.length });
    } catch (error: any) {
        console.error('Error fetching notes:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

export async function POST(request: NextRequest) {
    try {
        const body = await request.json();
        const {
            listId,
            title,
            description,
            done,
            important,
            flagged,
            dueDate,
            reminderDate,
            reminderTime,
            reminderRecurrence,
            subtasks,
            tags,
            attachments,
        } = body;

        if (!title) {
            return NextResponse.json(
                { error: 'Title is required' },
                { status: 400 }
            );
        }

        const db = await getNotesDatabase();
        const now = new Date().toISOString();
        const id = generateId();

        const note: Note = {
            id,
            listId: listId || null,
            title,
            description: description || '',
            done: done || false,
            important: important || false,
            flagged: flagged || false,
            dueDate: dueDate || null,
            reminderDate: reminderDate || null,
            reminderTime: reminderTime || null,
            reminderRecurrence: reminderRecurrence || 'none',
            createdAt: now,
            updatedAt: now,
            subtasks: subtasks || [],
            tags: tags || [],
            attachments: attachments || [],
        };

        await db.run(
            `INSERT INTO notes (
        id, list_id, title, description, done, important, flagged,
        due_date, reminder_date, reminder_time, reminder_recurrence, created_at, updated_at, subtasks, tags, attachments
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                note.id,
                note.listId,
                note.title,
                note.description,
                note.done ? 1 : 0,
                note.important ? 1 : 0,
                note.flagged ? 1 : 0,
                note.dueDate,
                note.reminderDate,
                note.reminderTime,
                note.reminderRecurrence,
                note.createdAt,
                note.updatedAt,
                JSON.stringify(note.subtasks),
                JSON.stringify(note.tags),
                JSON.stringify(note.attachments),
            ]
        );

        return NextResponse.json({ note }, { status: 201 });
    } catch (error: any) {
        console.error('Error creating note:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

