/**
 * API Route: Lists
 * GET /api/notes/lists - קבלת כל הרשימות
 * POST /api/notes/lists - יצירת רשימה חדשה
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';
import { List } from '@/types/notes';
import { generateId } from '@/src/lib/notes-helpers';

export async function GET() {
    try {
        const db = await getNotesDatabase();

        const lists = await db.all<List[]>(`
      SELECT * FROM lists 
      ORDER BY sort_order ASC, created_at DESC
    `);

        return NextResponse.json({ lists });
    } catch (error: any) {
        console.error('Error fetching lists:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

export async function POST(request: NextRequest) {
    try {
        const body = await request.json();
        const { name, color, icon, sortOrder } = body;

        if (!name) {
            return NextResponse.json(
                { error: 'Name is required' },
                { status: 400 }
            );
        }

        const db = await getNotesDatabase();
        const now = new Date().toISOString();
        const id = generateId();

        const list: List = {
            id,
            name,
            color: color || null,
            icon: icon || null,
            sortOrder: sortOrder || 0,
            createdAt: now,
            updatedAt: now,
        };

        await db.run(
            `INSERT INTO lists (id, name, color, icon, sort_order, created_at, updated_at)
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
            [list.id, list.name, list.color, list.icon, list.sortOrder, list.createdAt, list.updatedAt]
        );

        return NextResponse.json({ list }, { status: 201 });
    } catch (error: any) {
        console.error('Error creating list:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

