/**
 * API Route: Import Notes
 * POST /api/notes/import - ייבוא פתקים מ-CSV/XML
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';

export async function POST(request: NextRequest) {
    try {
        const formData = await request.formData();
        const file = formData.get('file') as File;

        if (!file) {
            return NextResponse.json(
                { error: 'לא נבחר קובץ' },
                { status: 400 }
            );
        }

        const content = await file.text();
        const format = file.name.endsWith('.xml') ? 'xml' : 'csv';

        const db = await getNotesDatabase();
        const now = new Date().toISOString();

        let imported = 0;

        if (format === 'xml') {
            // Parse XML
            const parser = new DOMParser();
            const xmlDoc = parser.parseFromString(content, 'text/xml');
            const notes = xmlDoc.getElementsByTagName('note');

            for (let i = 0; i < notes.length; i++) {
                const note = notes[i];
                const title = note.getElementsByTagName('title')[0]?.textContent || '';
                const description = note.getElementsByTagName('description')[0]?.textContent || '';
                const listName = note.getElementsByTagName('list')[0]?.textContent || '';
                const done = note.getElementsByTagName('done')[0]?.textContent === 'true' ? 1 : 0;
                const important = note.getElementsByTagName('important')[0]?.textContent === 'true' ? 1 : 0;
                const flagged = note.getElementsByTagName('flagged')[0]?.textContent === 'true' ? 1 : 0;
                const dueDate = note.getElementsByTagName('due_date')[0]?.textContent || null;

                if (title) {
                    // Find or create list
                    let listId = null;
                    if (listName && listName !== 'Inbox') {
                        const list = await db.get('SELECT id FROM lists WHERE name = ?', [listName]);
                        if (list) {
                            listId = list.id;
                        } else {
                            listId = `list_${Date.now()}_${i}`;
                            await db.run(
                                'INSERT INTO lists (id, name, color, created_at, updated_at) VALUES (?, ?, ?, ?, ?)',
                                [listId, listName, '#6366F1', now, now]
                            );
                        }
                    }

                    const noteId = `note_import_${Date.now()}_${i}`;
                    await db.run(
                        `INSERT INTO notes (
                            id, list_id, title, description, done, important, flagged,
                            due_date, reminder_date, reminder_time, reminder_recurrence,
                            created_at, updated_at, subtasks, tags, attachments
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NULL, NULL, 'none', ?, ?, '[]', '[]', '[]')`,
                        [noteId, listId, title, description, done, important, flagged, dueDate, now, now]
                    );
                    imported++;
                }
            }
        } else {
            // Parse CSV
            const lines = content.split('\n');
            const headers = lines[0].split(',');

            for (let i = 1; i < lines.length; i++) {
                const line = lines[i].trim();
                if (!line) continue;

                const values = parseCSVLine(line);
                if (values.length < 2) continue;

                const title = values[1] || '';
                const description = values[2] || '';
                const listName = values[3] || '';
                const done = values[4] === 'כן' ? 1 : 0;
                const important = values[5] === 'כן' ? 1 : 0;
                const flagged = values[6] === 'כן' ? 1 : 0;
                const dueDate = values[7] || null;

                if (title) {
                    // Find or create list
                    let listId = null;
                    if (listName && listName !== 'Inbox') {
                        const list = await db.get('SELECT id FROM lists WHERE name = ?', [listName]);
                        if (list) {
                            listId = list.id;
                        } else {
                            listId = `list_${Date.now()}_${i}`;
                            await db.run(
                                'INSERT INTO lists (id, name, color, created_at, updated_at) VALUES (?, ?, ?, ?, ?)',
                                [listId, listName, '#6366F1', now, now]
                            );
                        }
                    }

                    const noteId = `note_import_${Date.now()}_${i}`;
                    await db.run(
                        `INSERT INTO notes (
                            id, list_id, title, description, done, important, flagged,
                            due_date, reminder_date, reminder_time, reminder_recurrence,
                            created_at, updated_at, subtasks, tags, attachments
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NULL, NULL, 'none', ?, ?, '[]', '[]', '[]')`,
                        [noteId, listId, title, description, done, important, flagged, dueDate, now, now]
                    );
                    imported++;
                }
            }
        }

        const totalNotes = await db.get('SELECT COUNT(*) as count FROM notes');

        return NextResponse.json({
            success: true,
            message: `ייובאו ${imported} פתקים בהצלחה!`,
            stats: {
                imported,
                total: totalNotes.count
            }
        });

    } catch (error: any) {
        console.error('Error importing notes:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

function parseCSVLine(line: string): string[] {
    const result = [];
    let current = '';
    let inQuotes = false;

    for (let i = 0; i < line.length; i++) {
        const char = line[i];

        if (char === '"') {
            if (inQuotes && line[i + 1] === '"') {
                current += '"';
                i++;
            } else {
                inQuotes = !inQuotes;
            }
        } else if (char === ',' && !inQuotes) {
            result.push(current);
            current = '';
        } else {
            current += char;
        }
    }

    result.push(current);
    return result;
}

