/**
 * API Route: Import Real Notes from Images
 * POST /api/notes/import-real - ייבוא הפתקים האמיתיים מהתמונות
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';

export async function POST(request: NextRequest) {
    try {
        const db = await getNotesDatabase();
        const now = new Date().toISOString();

        console.log('📦 מייבא פתקים אמיתיים מהתמונות...');

        // נתוני הרשימות
        const lists = [
            { id: 'list_hashuv', name: 'חשוב', color: '#FF5733' },
            { id: 'list_dahuf', name: 'דחוף', color: '#FF6B00' },
            { id: 'list_domains', name: 'דומיינים חידוש', color: '#FFD700' },
            { id: 'list_money_2025', name: 'חייבים כסף 2025', color: '#32CD32' },
        ];

        // יצירת רשימות
        for (const list of lists) {
            await db.run(
                `INSERT OR REPLACE INTO lists (id, name, color, created_at, updated_at)
                 VALUES (?, ?, ?, ?, ?)`,
                [list.id, list.name, list.color, now, now]
            );
        }

        console.log(`✅ נוצרו ${lists.length} רשימות`);

        // הפתקים האמיתיים מהתמונות
        const notes = [
            // חשוב (17 פתקים - כולם חשובים)
            { list_id: 'list_hashuv', title: 'של אבי יהודה לוחמת יעם טובים', important: 1 },
            { list_id: 'list_hashuv', title: 'להעלות את התמונות של אביך רחלת דויו', important: 1 },
            { list_id: 'list_hashuv', title: 'לטפאך דעל במטרים', important: 1 },
            { list_id: 'list_hashuv', title: 'לכלן לשמוע עוזן כוכב רבתת', important: 1 },
            { list_id: 'list_hashuv', title: 'למסור את הקלפנית ותהומתות מאשי איתה', important: 1 },
            { list_id: 'list_hashuv', title: 'לומחוק חידועים מקרומליפים לומור אמום תהומתאות שאשי איתה', important: 1 },
            { list_id: 'list_hashuv', title: 'אגדל לחי יכרו', important: 1 },
            { list_id: 'list_hashuv', title: 'לטפכוך לשמוע חט אלקים והשמן באדון עם פרידעיות זרדיקות', important: 1 },
            { list_id: 'list_hashuv', title: 'צרך להסביר את האממים למאילו של מאוסרי לאך ליבטבך', important: 1 },
            { list_id: 'list_hashuv', title: 'פירק - בלעלא אליסונע שודע פיע ארץ https://kadosh/law.com?ת', important: 1 },
            { list_id: 'list_hashuv', title: 'חויר ציאט איחמיס זנלבי חמצת שדידואים זאנף חירד מח נוע למכני nw2000', important: 1 },
            { list_id: 'list_hashuv', title: 'מע מה האבבי השיואשיס שו חוברי מירושלים', important: 1 },
            { list_id: 'list_hashuv', title: 'לבדוק את הדיאעשים לי וידיען של אילה אחנה של משה', important: 1 },
            { list_id: 'list_hashuv', title: 'משאש לרע כבטבו את השחטישם לי האולקר החרדוש שילום אטסגון', important: 1 },
            { list_id: 'list_hashuv', title: 'צרך למסחר מאול לוטאמוס להטבילך באשאנן אשתח של הי הממקוק אין מודיע לרי', important: 1 },
            { list_id: 'list_hashuv', title: 'צרך לוודעש שסל משה ואסף חון משלם אבי להילהקב על פרידעית רודאעיק איעשים סאפין', important: 1 },
            { list_id: 'list_hashuv', title: 'עב הכוגן המזור צדיק ע לשלט 120 עם חי דחידעים', important: 1 },

            // חייבים כסף 2025 (16 פתקים)
            { list_id: 'list_money_2025', title: 'לטפכוך לשמוע חט אלקים והשמן באדון עם פרידעיות זרדיקות', important: 1 },
            { list_id: 'list_money_2025', title: 'דויר חיב של שלים איףום', important: 1 },
            { list_id: 'list_money_2025', title: 'חיים כמזה הכינו מלן פי חופה חרדים', important: 0 },
            { list_id: 'list_money_2025', title: 'יחיום גם:נכפדי: צרייף לשלח חרדעים ואילה הפנת שעד של איסט סושר', important: 1 },
            { list_id: 'list_money_2025', title: 'צרך למאחד מאול לוטאמוס להטבילם באשאנן אשתח של הי הממקוק אין מודיע לרי', important: 1 },
            { list_id: 'list_money_2025', title: 'פירק - בלעלא אליסונע שודע פיע ארץ https://kadosh/law.com?', important: 1 },
            { list_id: 'list_money_2025', title: 'לבדוק את החרדושם לי וידיען של אילה איטא שעף', important: 1 },
            { list_id: 'list_money_2025', title: 'צרך לחדוש שסל משה ואסף חון משלם אבי להילהקב על פרידעית רודאעיק איעשים', important: 1 },
            { list_id: 'list_money_2025', title: 'עב הכוגן המזור צדיק ע לשלט 120 עם פרי דחידעים', important: 1 },
            { list_id: 'list_money_2025', title: 'דויר חיב של שלים איףום אחטון', important: 0 },
            { list_id: 'list_money_2025', title: 'שירה רדן פי חופא אחטון', important: 0 },
            { list_id: 'list_money_2025', title: 'נפיכו לין צוממה 2500 משי שעמ', important: 0 },
            { list_id: 'list_money_2025', title: 'שיר שעודה לין חון ניחמדו להוישו פישקטון', important: 0 },
            { list_id: 'list_money_2025', title: 'חייב כיפום (עבדי יאהדה שודע פיע חי כוקנו)', important: 0 },
            { list_id: 'list_money_2025', title: 'פוריס גרוב חרדעש אישון שעלש כשמו חרדוא מפטקום 1200', important: 0 },
            { list_id: 'list_money_2025', title: 'משרי כיתרווה שלים שגיכם דיקנעום', important: 0 },

            // דחוף (9 פתקים)
            { list_id: 'list_dahuf', title: 'פוקרבין חו לי כסו שלים שגיכם ואישגי אחטון', important: 0 },
            { list_id: 'list_dahuf', title: 'אנליזינית - איסד אחת הסטון לרכולא כן במדלום האורגיי של האנקר', important: 0 },
            { list_id: 'list_dahuf', title: 'לינומו 5000 שעקל לטינועניכפטלם', important: 0 },
            { list_id: 'list_dahuf', title: 'לינומו 50 שקל לספוטובריבושקיס', important: 0 },
            { list_id: 'list_dahuf', title: 'לבדוק את החידועים (אישראלוניס) שימה נוסנטר לו את האגני לטפון', important: 0 },
            { list_id: 'list_dahuf', title: 'מפוי כסניבות + 360 פעוכת ודיחיפים', important: 0 },
            { list_id: 'list_dahuf', title: 'לטפוע מופרי אישון שאלירים המשג שפרועה איריעים צרעוכם איםכו יב פי ויריגעו', important: 0 },
            { list_id: 'list_dahuf', title: 'את הפודריל [בן לין אחטון', important: 0 },
            { list_id: 'list_dahuf', title: 'מיםגפניס (ח× פי מח sonak · כעשב ح)', important: 0 },

            // דומיינים חידוש (5 פתקים)
            { list_id: 'list_domains', title: 'hob-law.co.il 07/10/2018 08/10/2025', important: 0 },
            { list_id: 'list_domains', title: 'hob-law.co.il 08/10/2025 19:59:59', important: 0 },
            { list_id: 'list_domains', title: 'limousine4u.co.il 08/09/2020 09/09/2025', important: 0 },
            { list_id: 'list_domains', title: 'balloon2u.co.il', important: 0 },
            { list_id: 'list_domains', title: 'bdsoap.co.il תפוגה 01:59:59 18/03/2025 בעל: marios steinfeld', important: 0 },
        ];

        let noteCount = 0;
        for (const note of notes) {
            const noteId = `note_real_${Date.now()}_${noteCount}`;
            await db.run(
                `INSERT INTO notes (
                    id, list_id, title, description, done, important, flagged,
                    due_date, reminder_date, reminder_time, reminder_recurrence,
                    created_at, updated_at, subtasks, tags, attachments
                ) VALUES (?, ?, ?, '', 0, ?, 0, NULL, NULL, NULL, 'none', ?, ?, '[]', '[]', '[]')`,
                [noteId, note.list_id, note.title, note.important, now, now]
            );
            noteCount++;
        }

        console.log(`✅ ייובאו ${noteCount} פתקים אמיתיים`);

        const listCount = await db.get('SELECT COUNT(*) as count FROM lists');
        const totalNotes = await db.get('SELECT COUNT(*) as count FROM notes');

        return NextResponse.json({
            success: true,
            message: 'הפתקים האמיתיים ייובאו בהצלחה! 🎉',
            stats: {
                lists: listCount.count,
                notes: totalNotes.count,
                imported: noteCount
            }
        }, { status: 200 });

    } catch (error: any) {
        console.error('Error importing real notes:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

