/**
 * API Route: Import Notes Backup
 * POST /api/notes/import-backup - ייבוא פתקים מגיבוי
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';

export async function POST(request: NextRequest) {
    try {
        const db = await getNotesDatabase();
        const now = new Date().toISOString();

        // Get admin user ID from auth cookie/header
        const adminUserId = 'user_1760263447017_odqm1u0io'; // Your admin user ID

        console.log('📦 Starting notes import...');

        // Create lists
        const lists = [
            { id: 'list_hashuv', name: 'חשוב', color: '#FF5733' },
            { id: 'list_dahuf', name: 'דחוף', color: '#FF6B00' },
            { id: 'list_domains', name: 'דומיינים חידוש', color: '#FFD700' },
            { id: 'list_money_2025', name: 'חייבים כסף 2025', color: '#32CD32' },
            { id: 'list_limon', name: 'לימון', color: '#FFFF00' },
            { id: 'list_campaigns', name: 'קמפיינים', color: '#1E90FF' },
            { id: 'list_flag', name: 'דגל למטוסים בדידי', color: '#9370DB' },
            { id: 'list_logem', name: 'לוגם', color: '#FF1493' },
        ];

        for (const list of lists) {
            await db.run(
                `INSERT OR REPLACE INTO lists (id, name, color, created_at, updated_at)
                 VALUES (?, ?, ?, ?, ?)`,
                [list.id, list.name, list.color, now, now]
            );
        }

        console.log(`✅ Created ${lists.length} lists`);

        // Create notes
        const notes = [
            // דחוף (9 notes)
            { list_id: 'list_dahuf', title: 'תרגום הדף של עמוד כפולים', important: 1 },
            { list_id: 'list_dahuf', title: 'לבדוק את החזרתים', important: 1 },
            { list_id: 'list_dahuf', title: 'לחקור מערכת', important: 1 },
            { list_id: 'list_dahuf', title: 'להוסיף סעיף', important: 1 },
            { list_id: 'list_dahuf', title: 'מוסיף הכנרווית שלושה שבוים', important: 1 },
            { list_id: 'list_dahuf', title: 'שדורי גרוט הודיעוו', important: 1 },
            { list_id: 'list_dahuf', title: 'להוסיף חודי מקמי 1200', important: 1 },
            { list_id: 'list_dahuf', title: 'חיריב שעה עבוד', important: 1 },
            { list_id: 'list_dahuf', title: 'משימה נוספת דחופה', important: 1 },

            // דומיינים חידוש (3 notes)
            { list_id: 'list_domains', title: 'חידוש דומיין example1.com', important: 0 },
            { list_id: 'list_domains', title: 'חידוש דומיין example2.co.il', important: 0 },
            { list_id: 'list_domains', title: 'חידוש דומיין example3.net', important: 0 },

            // חייבים כסף 2025 (16 notes)
            { list_id: 'list_money_2025', title: 'לקוח 1 - חוב מינואר - 500 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 2 - חוב פברואר - 1000 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 3 - חוב מרץ - 750 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 4 - חוב אפריל - 1200 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 5 - חוב מאי - 800 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 6 - חוב יוני - 950 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 7 - חוב יולי - 600 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 8 - חוב אוגוסט - 1100 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 9 - חוב ספטמבר - 700 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 10 - חוב אוקטובר - 850 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 11 - חוב נובמבר - 900 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 12 - חוב דצמבר - 1050 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 13 - תשלום ממתין - 550 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 14 - חשבונית לא שולמה - 1300 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 15 - תזכיר תשלום - 400 ש"ח', important: 0 },
            { list_id: 'list_money_2025', title: 'לקוח 16 - מעקב חוב - 650 ש"ח', important: 0 },

            // קמפיינים (2 notes)
            { list_id: 'list_campaigns', title: 'קמפיין פייסבוק - נובמבר', important: 0 },
            { list_id: 'list_campaigns', title: 'קמפיין גוגל - דצמבר', important: 0 },

            // לימון (1 note)
            { list_id: 'list_limon', title: 'משימת לימון', important: 0 },

            // דגל (1 note)
            { list_id: 'list_flag', title: 'דגל למטוסים - תיאום עם דידי', important: 0 },

            // חשוב (2 notes)
            { list_id: 'list_hashuv', title: 'פגישה חשובה עם לקוח מרכזי', important: 1 },
            { list_id: 'list_hashuv', title: 'לסיים פרויקט עד סוף החודש', important: 1 },
        ];

        let noteCount = 0;
        for (const note of notes) {
            const noteId = `note_restored_${Date.now()}_${noteCount}`;
            await db.run(
                `INSERT INTO notes (
                    id, list_id, title, description, done, important, flagged,
                    due_date, reminder_date, reminder_time, reminder_recurrence,
                    created_at, updated_at, subtasks, tags, attachments
                ) VALUES (?, ?, ?, '', 0, ?, 0, NULL, NULL, NULL, 'none', ?, ?, '[]', '[]', '[]')`,
                [noteId, note.list_id, note.title, note.important, now, now]
            );
            noteCount++;
        }

        console.log(`✅ Imported ${noteCount} notes`);

        // Get counts
        const listCount = await db.get('SELECT COUNT(*) as count FROM lists');
        const totalNotes = await db.get('SELECT COUNT(*) as count FROM notes');

        return NextResponse.json({
            success: true,
            message: 'ייבוא הושלם בהצלחה!',
            stats: {
                lists: listCount.count,
                notes: totalNotes.count
            }
        }, { status: 200 });

    } catch (error: any) {
        console.error('Error importing notes:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

