/**
 * API Route: Export Notes
 * GET /api/notes/export?format=csv|xml - ייצוא פתקים
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';

export async function GET(request: NextRequest) {
    try {
        const { searchParams } = new URL(request.url);
        const format = searchParams.get('format') || 'csv';

        const db = await getNotesDatabase();

        // Get all notes with list names
        const notes = await db.all(`
            SELECT 
                n.*,
                l.name as list_name
            FROM notes n
            LEFT JOIN lists l ON n.list_id = l.id
            ORDER BY n.created_at DESC
        `);

        if (format === 'xml') {
            // Export as XML
            let xml = '<?xml version="1.0" encoding="UTF-8"?>\n<notes>\n';

            notes.forEach((note: any) => {
                xml += '  <note>\n';
                xml += `    <id>${escapeXml(note.id)}</id>\n`;
                xml += `    <title>${escapeXml(note.title)}</title>\n`;
                xml += `    <description>${escapeXml(note.description || '')}</description>\n`;
                xml += `    <list>${escapeXml(note.list_name || 'Inbox')}</list>\n`;
                xml += `    <done>${note.done ? 'true' : 'false'}</done>\n`;
                xml += `    <important>${note.important ? 'true' : 'false'}</important>\n`;
                xml += `    <flagged>${note.flagged ? 'true' : 'false'}</flagged>\n`;
                xml += `    <due_date>${escapeXml(note.due_date || '')}</due_date>\n`;
                xml += `    <created_at>${escapeXml(note.created_at)}</created_at>\n`;
                xml += '  </note>\n';
            });

            xml += '</notes>';

            return new NextResponse(xml, {
                headers: {
                    'Content-Type': 'application/xml',
                    'Content-Disposition': `attachment; filename="notes-${new Date().toISOString().split('T')[0]}.xml"`,
                },
            });
        } else {
            // Export as CSV
            const csv = [
                // Header
                ['ID', 'כותרת', 'תיאור', 'רשימה', 'בוצע', 'חשוב', 'דגל', 'תאריך יעד', 'נוצר בתאריך'].join(','),
                // Data
                ...notes.map((note: any) => [
                    escapeCsv(note.id),
                    escapeCsv(note.title),
                    escapeCsv(note.description || ''),
                    escapeCsv(note.list_name || 'Inbox'),
                    note.done ? 'כן' : 'לא',
                    note.important ? 'כן' : 'לא',
                    note.flagged ? 'כן' : 'לא',
                    escapeCsv(note.due_date || ''),
                    escapeCsv(note.created_at),
                ].join(','))
            ].join('\n');

            // Add BOM for Excel Hebrew support
            const bom = '\uFEFF';

            return new NextResponse(bom + csv, {
                headers: {
                    'Content-Type': 'text/csv; charset=utf-8',
                    'Content-Disposition': `attachment; filename="notes-${new Date().toISOString().split('T')[0]}.csv"`,
                },
            });
        }
    } catch (error: any) {
        console.error('Error exporting notes:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

function escapeCsv(value: string): string {
    if (!value) return '';
    const str = String(value);
    if (str.includes(',') || str.includes('"') || str.includes('\n')) {
        return `"${str.replace(/"/g, '""')}"`;
    }
    return str;
}

function escapeXml(value: string): string {
    if (!value) return '';
    return String(value)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&apos;');
}

