import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

// GET /api/events - קבלת כל האירועים
export async function GET(request: NextRequest) {
    const db = getDb();
    try {
        const { searchParams } = new URL(request.url);
        const customer_id = searchParams.get('customer_id');
        const event_type = searchParams.get('event_type');
        const status = searchParams.get('status');
        const start_date = searchParams.get('start_date');
        const end_date = searchParams.get('end_date');
        const limit = searchParams.get('limit') || '50';
        const offset = searchParams.get('offset') || '0';

        let query = `
            SELECT 
                e.*,
                c.name as customer_name,
                c.company as customer_company,
                c.email as customer_email,
                c.phone as customer_phone,
                d.name as domain_name,
                p.name as project_title
            FROM events e
            LEFT JOIN customers c ON e.customer_id = c.id
            LEFT JOIN domains d ON e.domain_id = d.id
            LEFT JOIN projects p ON e.project_id = p.id
            WHERE 1=1
        `;

        const params: any[] = [];

        if (customer_id) {
            query += ' AND e.customer_id = ?';
            params.push(customer_id);
        }

        if (event_type) {
            query += ' AND e.event_type = ?';
            params.push(event_type);
        }

        if (status) {
            query += ' AND e.status = ?';
            params.push(status);
        }

        if (start_date) {
            query += ' AND date(e.start_datetime) >= ?';
            params.push(start_date);
        }

        if (end_date) {
            query += ' AND date(e.start_datetime) <= ?';
            params.push(end_date);
        }

        query += ' ORDER BY e.start_datetime ASC';
        query += ' LIMIT ? OFFSET ?';
        params.push(parseInt(limit), parseInt(offset));

        const events = db.prepare(query).all(...params) as unknown as any[];

        // ספירת סה"כ
        let countQuery = `
            SELECT COUNT(*) as total
            FROM events e
            WHERE 1=1
        `;
        const countParams: any[] = [];

        if (customer_id) {
            countQuery += ' AND e.customer_id = ?';
            countParams.push(customer_id);
        }

        if (event_type) {
            countQuery += ' AND e.event_type = ?';
            countParams.push(event_type);
        }

        if (status) {
            countQuery += ' AND e.status = ?';
            countParams.push(status);
        }

        if (start_date) {
            countQuery += ' AND date(e.start_datetime) >= ?';
            countParams.push(start_date);
        }

        if (end_date) {
            countQuery += ' AND date(e.start_datetime) <= ?';
            countParams.push(end_date);
        }

        const totalResult = db.prepare(countQuery).get(...countParams) as unknown as { total: number };

        return NextResponse.json({
            events,
            total: totalResult.total,
            limit: parseInt(limit),
            offset: parseInt(offset)
        });

    } catch (e: unknown) {
        console.error('Get events error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת אירועים: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}

// POST /api/events - יצירת אירוע חדש
export async function POST(request: NextRequest) {
    const db = getDb();
    try {
        const body = await request.json();

        // וולידציה בסיסית
        if (!body.title) {
            return NextResponse.json(
                { error: 'כותרת האירוע היא חובה' },
                { status: 400 }
            );
        }

        if (!body.start_datetime) {
            return NextResponse.json(
                { error: 'תאריך ושעת התחלה הם חובה' },
                { status: 400 }
            );
        }

        // וולידציה: אם יש customer_id, לוודא שהוא קיים
        if (body.customer_id) {
            const customer = db.prepare('SELECT id FROM customers WHERE id = ?').get(body.customer_id) as unknown;
            if (!customer) {
                return NextResponse.json(
                    { error: 'לקוח לא נמצא' },
                    { status: 400 }
                );
            }
        }

        // Get domain info if needed
        if (body.domain_id) {
            const domain = db.prepare('SELECT customer_id FROM domains WHERE id = ?').get(body.domain_id) as unknown;
            if (!domain) {
                return NextResponse.json(
                    { error: 'דומיין לא נמצא' },
                    { status: 400 }
                );
            }
            if (body.customer_id && (domain as { customer_id: string }).customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הדומיין לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        // Get project info if needed
        if (body.project_id) {
            const project = db.prepare('SELECT customer_id FROM projects WHERE id = ?').get(body.project_id) as unknown;
            if (!project) {
                return NextResponse.json(
                    { error: 'פרויקט לא נמצא' },
                    { status: 400 }
                );
            }
            if (body.customer_id && (project as { customer_id: string }).customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הפרויקט לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        const insertQuery = `
            INSERT INTO events (
                title, description, start_datetime, end_datetime, location,
                customer_id, domain_id, project_id, quote_id, task_id,
                event_type, status, attendees, reminder_minutes, created_by, tags
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `;

        const result = db.prepare(insertQuery).run(
            body.title,
            body.description || null,
            body.start_datetime,
            body.end_datetime || null,
            body.location || null,
            body.customer_id || null,
            body.domain_id || null,
            body.project_id || null,
            body.quote_id || null,
            body.task_id || null,
            body.event_type || 'meeting',
            body.status || 'scheduled',
            body.attendees ? JSON.stringify(body.attendees) : null,
            body.reminder_minutes || 15,
            body.created_by || null,
            body.tags ? JSON.stringify(body.tags) : null
        );

        // Get the created event
        const newEvent = db.prepare(`
            SELECT 
                e.*,
                c.name as customer_name,
                c.company as customer_company,
                d.name as domain_name,
                p.name as project_title
            FROM events e
            LEFT JOIN customers c ON e.customer_id = c.id
            LEFT JOIN domains d ON e.domain_id = d.id
            LEFT JOIN projects p ON e.project_id = p.id
            WHERE e.id = ?
        `).get(result.lastInsertRowid) as unknown;

        return NextResponse.json(newEvent, { status: 201 });

    } catch (e: unknown) {
        console.error('Create event error:', e);
        return NextResponse.json({ error: 'שגיאה ביצירת אירוע: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}
