/**
 * API Route: Initialize Events Table
 * POST /api/events/init - יצירת טבלת אירועים
 */

import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

export async function POST(request: NextRequest) {
    try {
        const dbPath = path.join(process.cwd(), 'comprehensive.db');
        const db = new Database(dbPath);

        console.log('🔧 Creating events table...');

        // Drop the table if it exists with wrong schema
        db.exec(`DROP TABLE IF EXISTS events;`);

        // Create events table with correct schema
        db.exec(`
            CREATE TABLE events (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                description TEXT,
                start_datetime TEXT NOT NULL,
                end_datetime TEXT,
                location TEXT,
                customer_id TEXT,
                domain_id TEXT,
                project_id TEXT,
                quote_id TEXT,
                task_id TEXT,
                event_type TEXT DEFAULT 'meeting',
                status TEXT DEFAULT 'scheduled',
                attendees TEXT,
                reminder_minutes INTEGER DEFAULT 15,
                created_by TEXT,
                tags TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES customers(id),
                FOREIGN KEY (domain_id) REFERENCES domains(id),
                FOREIGN KEY (project_id) REFERENCES projects(id),
                FOREIGN KEY (task_id) REFERENCES tasks(id)
            );
        `);

        // Create indexes
        db.exec(`
            CREATE INDEX IF NOT EXISTS idx_events_customer ON events(customer_id);
            CREATE INDEX IF NOT EXISTS idx_events_project ON events(project_id);
            CREATE INDEX IF NOT EXISTS idx_events_start ON events(start_datetime);
            CREATE INDEX IF NOT EXISTS idx_events_type ON events(event_type);
            CREATE INDEX IF NOT EXISTS idx_events_status ON events(status);
        `);

        // Count events
        const count = db.prepare('SELECT COUNT(*) as count FROM events').get() as { count: number };

        db.close();

        return NextResponse.json({
            success: true,
            message: 'טבלת האירועים נוצרה בהצלחה!',
            stats: {
                events: count.count
            }
        });

    } catch (error: any) {
        console.error('Error initializing events table:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

