import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

// GET /api/event-tags - Get all unique tags from events
export async function GET() {
    const db = getDb();
    try {
        // Get all events with tags
        const events = db.prepare('SELECT tags FROM events WHERE tags IS NOT NULL').all() as { tags: string }[];

        // Extract all unique tags
        const tagsMap = new Map<string, { id: string; name: string; color: string; description?: string }>();

        events.forEach(event => {
            try {
                const eventTags = JSON.parse(event.tags);
                if (Array.isArray(eventTags)) {
                    eventTags.forEach(tag => {
                        if (!tagsMap.has(tag.id)) {
                            tagsMap.set(tag.id, tag);
                        }
                    });
                }
            } catch (e) {
                // Skip invalid JSON
            }
        });

        const tags = Array.from(tagsMap.values()).sort((a, b) => a.name.localeCompare(b.name));
        return NextResponse.json(tags);
    } catch (e: unknown) {
        console.error('List event tags error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת התגיות: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}

// POST /api/event-tags - Create a new tag (returns the tag object for immediate use)
export async function POST(request: NextRequest) {
    try {
        const body = await request.json();

        if (!body.name || !body.color) {
            return NextResponse.json({ error: 'שם וצבע התגית הם שדות חובה' }, { status: 400 });
        }

        // Create a new tag object (no DB storage, just return for client use)
        const newTag = {
            id: `tag_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
            name: body.name,
            color: body.color,
            description: body.description || undefined
        };

        return NextResponse.json(newTag, { status: 201 });
    } catch (e: unknown) {
        console.error('Create event tag error:', e);
        return NextResponse.json({ error: 'שגיאה ביצירת התגית: ' + (e as Error).message }, { status: 500 });
    }
}
