import { NextRequest, NextResponse } from 'next/server';
import { sendEmail, sendBulkEmails, populateTemplate } from '@/src/lib/email/cpanel-mailer';

export async function POST(request: NextRequest) {
    try {
        const body = await request.json();
        const { type, data } = body;

        if (type === 'single') {
            const { to, subject, html, text } = data;
            if (!to || !subject || !html) {
                return NextResponse.json({ error: 'Missing required fields' }, { status: 400 });
            }
            const result = await sendEmail({ to, subject, html, text });
            return NextResponse.json(result);
        }

        if (type === 'bulk') {
            const { recipients, templateSubject, templateBody } = data;
            if (!recipients || !Array.isArray(recipients)) {
                return NextResponse.json({ error: 'Invalid recipients' }, { status: 400 });
            }

            const emails = recipients.map(r => ({
                to: r.email,
                subject: populateTemplate(templateSubject, {
                    customerName: r.name,
                    domain: r.domain,
                    amount: r.amount.toString(),
                    daysOverdue: r.daysOverdue.toString(),
                    packageName: r.packageName || '',
                    companyName: r.companyName || '',
                    paymentLink: r.paymentLink || '',
                    supportEmail: process.env.SMTP_FROM_EMAIL || 'sales@tsm.co.il',
                }),
                html: populateTemplate(templateBody, {
                    customerName: r.name,
                    domain: r.domain,
                    amount: r.amount.toString(),
                    daysOverdue: r.daysOverdue.toString(),
                    packageName: r.packageName || '',
                    companyName: r.companyName || '',
                    paymentLink: r.paymentLink || '',
                    supportEmail: process.env.SMTP_FROM_EMAIL || 'sales@tsm.co.il',
                }),
            }));

            const results = await sendBulkEmails(emails);
            const successful = results.filter(r => r.success).length;
            const failed = results.filter(r => !r.success).length;

            return NextResponse.json({
                success: true,
                message: `נשלחו ${successful} אימיילים`,
                stats: { total: results.length, successful, failed },
                results,
            });
        }

        return NextResponse.json({ error: 'Invalid request type' }, { status: 400 });
    } catch (error) {
        console.error('Email API Error:', error);
        return NextResponse.json(
            { error: 'Server error', details: error instanceof Error ? error.message : 'Unknown' },
            { status: 500 }
        );
    }
}

export async function GET(request: NextRequest) {
    try {
        const { searchParams } = new URL(request.url);
        const testUser = searchParams.get('testUser');
        const testPass = searchParams.get('testPass');
        const testPort = searchParams.get('testPort');

        // אם יש פרמטרים לבדיקה, נשתמש בהם
        if (testUser && testPass) {
            const nodemailer = require('nodemailer');
            const port = testPort ? parseInt(testPort) : 587;
            const testTransporter = nodemailer.createTransport({
                host: 'mail.tsm.co.il',
                port: port,
                secure: port === 465, // SSL only for port 465
                auth: {
                    user: testUser,
                    pass: testPass,
                },
                // Don't force TLS - let the server decide
                connectionTimeout: 10000,
                greetingTimeout: 10000,
            });

            await testTransporter.verify();
            return NextResponse.json({
                success: true,
                message: `✅ התחברות הצליחה עם ${testUser} על פורט ${testPort || 587}!`
            });
        }

        const { testEmailConnection } = await import('@/src/lib/email/cpanel-mailer');
        const result = await testEmailConnection();
        return NextResponse.json(result);
    } catch (error: any) {
        return NextResponse.json(
            {
                success: false,
                error: error.message || 'Unknown error',
                code: error.code,
                response: error.response
            }
        );
    }
}
