import { NextRequest, NextResponse } from 'next/server';
import { openDB } from '@/src/lib/db';
import sqlite3 from 'sqlite3';
import path from 'path';

export async function GET(request: NextRequest) {
    try {
        const db = await openDB();
        // Also open domains.db to include customers referenced there (domains/projects and possibly other modules)
        const domainsDbPath = path.join(process.cwd(), 'domains.db');
        const domainsDb = await new Promise<sqlite3.Database>((resolve) => {
            const database = new sqlite3.Database(domainsDbPath, () => resolve(database));
        });

        // 1) Fetch existing customers from both databases
        const [existingCustomersComp, existingCustomersDomains] = await Promise.all([
            new Promise<any[]>((resolve, reject) => {
                db.all(
                    `SELECT id, name, email, phone, company, status, stage, value, assigned_sales_rep, notes
                     FROM customers`,
                    (err, rows) => {
                        if (err) reject(err); else resolve(rows || []);
                    }
                );
            }),
            new Promise<any[]>((resolve, reject) => {
                domainsDb.all(
                    `SELECT id, name, email, phone, company, status, stage, value, assigned_sales_rep, notes
                     FROM customers`,
                    (err, rows) => {
                        if (err) reject(err); else resolve(rows || []);
                    }
                );
            })
        ]);

        // Merge customers from both databases, prioritizing domains.db
        const existingCustomers = [...existingCustomersDomains, ...existingCustomersComp.filter(c =>
            !existingCustomersDomains.some(d => d.id === c.id)
        )];

        // 2) Collect distinct customer_ids referenced in other modules (from both comprehensive.db and domains.db)
        const [domainIdsComp, projectIdsComp, domainIdsDomainsDb, projectIdsDomainsDb] = await Promise.all([
            new Promise<string[]>((resolve) => {
                db.all(`SELECT DISTINCT customer_id AS id FROM domains`, (err, rows) => {
                    if (err) return resolve([]);
                    resolve((rows || []).map((r: any) => r.id));
                });
            }),
            new Promise<string[]>((resolve) => {
                db.all(`SELECT DISTINCT customer_id AS id FROM projects`, (err, rows) => {
                    if (err) return resolve([]);
                    resolve((rows || []).map((r: any) => r.id));
                });
            }),
            new Promise<string[]>((resolve) => {
                domainsDb.all(`SELECT DISTINCT customer_id AS id FROM domains`, (err, rows) => {
                    if (err) return resolve([]);
                    resolve((rows || []).map((r: any) => r.id));
                });
            }),
            new Promise<string[]>((resolve) => {
                domainsDb.all(`SELECT DISTINCT customer_id AS id FROM projects`, (err, rows) => {
                    if (err) return resolve([]);
                    resolve((rows || []).map((r: any) => r.id));
                });
            })
        ]);

        // 3) Find referenced customers missing from customers table
        const referencedIds = new Set<string>([
            ...domainIdsComp,
            ...projectIdsComp,
            ...domainIdsDomainsDb,
            ...projectIdsDomainsDb
        ].filter(Boolean));

        const presentIds = new Set<string>(existingCustomers.map(c => c.id));

        const syntheticCustomers: any[] = [];
        for (const id of referencedIds) {
            if (!presentIds.has(id)) {
                // Minimal placeholder so it appears in the list
                syntheticCustomers.push({
                    id,
                    name: id, // עד שממלאים פרטים
                    email: '',
                    phone: '',
                    company: '',
                    status: 'lead',
                    stage: 'lead',
                    value: 0,
                    assigned_sales_rep: '',
                    notes: 'נוצר אוטומטית מרפרנס במודולים'
                });
            }
        }

        // 4) Merge and sort by name (Hebrew-friendly simple sort)
        const allCustomers = [...existingCustomers, ...syntheticCustomers].sort((a, b) => {
            return (a.name || '').localeCompare(b.name || '', 'he');
        });

        return NextResponse.json(allCustomers);
    } catch (error) {
        console.error('Error fetching customers:', error);
        return NextResponse.json(
            { error: 'Failed to fetch customers' },
            { status: 500 }
        );
    }
}

export async function POST(request: NextRequest) {
    try {
        const body = await request.json();
        const db = await openDB();

        // Generate unique ID
        const id = `customer_${Date.now()}`;
        const now = new Date().toISOString();

        const result = await new Promise((resolve, reject) => {
            db.run(`
                INSERT INTO customers (
                    id, name, email, phone, company, status, stage, 
                    assigned_sales_rep, value, notes, created_date, last_activity, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `, [
                id,
                body.name,
                body.email,
                body.phone || '',
                body.company || '',
                body.status || 'lead',
                body.stage || 'quote',
                body.assigned_sales_rep || '',
                body.value || 0,
                body.notes || '',
                now,
                now,
                now
            ], function (err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({ id: this.lastID });
                }
            });
        });

        // Return the created customer
        const newCustomer = await new Promise((resolve, reject) => {
            db.get('SELECT * FROM customers WHERE id = ?', [id], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });

        return NextResponse.json(newCustomer, { status: 201 });
    } catch (error) {
        console.error('Error creating customer:', error);
        return NextResponse.json(
            { error: 'Failed to create customer' },
            { status: 500 }
        );
    }
}