/**
 * API Route: Check and Fix Customers Table
 * POST /api/customers/fix - תיקון טבלת לקוחות
 */

import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

export async function POST(request: NextRequest) {
    try {
        const dbPath = path.join(process.cwd(), 'comprehensive.db');
        const db = new Database(dbPath);

        console.log('🔧 Fixing customers table...');

        // Check if table exists
        const tableExists = db.prepare(`
            SELECT name FROM sqlite_master WHERE type='table' AND name='customers'
        `).get();

        if (!tableExists) {
            // Create customers table
            db.exec(`
                CREATE TABLE customers (
                    id TEXT PRIMARY KEY,
                    name TEXT NOT NULL,
                    email TEXT,
                    phone TEXT,
                    company TEXT,
                    status TEXT DEFAULT 'active',
                    stage TEXT,
                    value REAL,
                    assigned_sales_rep TEXT,
                    notes TEXT,
                    created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                    updated_at TEXT DEFAULT CURRENT_TIMESTAMP
                );
            `);
            console.log('✅ Customers table created');
        } else {
            // Get current columns
            const columns = db.prepare(`PRAGMA table_info(customers)`).all() as any[];
            const columnNames = columns.map(c => c.name);

            // Add missing columns
            const requiredColumns = [
                { name: 'company', type: 'TEXT', default: 'NULL' },
                { name: 'status', type: 'TEXT', default: "'active'" },
                { name: 'stage', type: 'TEXT', default: 'NULL' },
                { name: 'value', type: 'REAL', default: 'NULL' },
                { name: 'assigned_sales_rep', type: 'TEXT', default: 'NULL' },
                { name: 'notes', type: 'TEXT', default: 'NULL' },
                { name: 'created_at', type: 'TEXT', default: 'NULL' },
                { name: 'updated_at', type: 'TEXT', default: 'NULL' }
            ];

            for (const col of requiredColumns) {
                if (!columnNames.includes(col.name)) {
                    const sql = `ALTER TABLE customers ADD COLUMN ${col.name} ${col.type} DEFAULT ${col.default}`;
                    db.exec(sql);
                    console.log(`✅ Added column: ${col.name}`);
                }
            }

            // Update NULL timestamps to current time
            db.exec(`
                UPDATE customers 
                SET created_at = datetime('now') 
                WHERE created_at IS NULL
            `);
            db.exec(`
                UPDATE customers 
                SET updated_at = datetime('now') 
                WHERE updated_at IS NULL
            `);
        }

        // Create indexes
        db.exec(`
            CREATE INDEX IF NOT EXISTS idx_customers_email ON customers(email);
            CREATE INDEX IF NOT EXISTS idx_customers_status ON customers(status);
        `);

        // Count customers
        const count = db.prepare('SELECT COUNT(*) as count FROM customers').get() as { count: number };

        db.close();

        return NextResponse.json({
            success: true,
            message: 'טבלת הלקוחות תוקנה בהצלחה!',
            stats: {
                customers: count.count
            }
        });

    } catch (error: any) {
        console.error('Error fixing customers table:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

