import { NextRequest, NextResponse } from 'next/server';
import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { ComprehensiveStats } from '@/src/types/comprehensive';

export async function GET(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        // Customer statistics
        const customerStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
                SUM(CASE WHEN status = 'lead' THEN 1 ELSE 0 END) as leads,
                SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive
            FROM customers
        `).get() as unknown;

        // Domain statistics
        const domainStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
                SUM(CASE WHEN status = 'expired' THEN 1 ELSE 0 END) as expired,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending
            FROM domains
        `).get() as unknown;

        // Project statistics
        const projectStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as active,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'planning' THEN 1 ELSE 0 END) as planning
            FROM projects
        `).get() as unknown;

        // Quote statistics
        const quoteStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
            FROM quotes
        `).get() as unknown;

        // Payment statistics
        const paymentStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                COALESCE(SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END), 0) as revenue
            FROM payments
        `).get() as unknown;

        // Activity statistics
        const activityStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN due_date < datetime('now') AND status = 'pending' THEN 1 ELSE 0 END) as overdue
            FROM activities
        `).get() as unknown;

        // Reminder statistics
        const reminderStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN reminder_date < datetime('now') AND status = 'pending' THEN 1 ELSE 0 END) as overdue,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed
            FROM reminders
        `).get() as unknown;

        // Service statistics
        const emailServiceStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active
            FROM email_services
        `).get() as unknown;

        const technicalServiceStats = db.prepare(`
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active
            FROM technical_services
        `).get() as unknown;

        // Calculate overdue projects (projects past end_date)
        const overdueProjects = db.prepare(`
            SELECT COUNT(*) as count
            FROM projects 
            WHERE end_date < date('now') AND status IN ('planning', 'in_progress')
        `).get() as unknown;

        // Calculate overdue payments (payments past due_date)
        const overduePayments = db.prepare(`
            SELECT COUNT(*) as count
            FROM payments 
            WHERE due_date < datetime('now') AND status = 'pending'
        `).get() as unknown;

        // Calculate expiring domains (next payment within 30 days)
        const expiringDomains = db.prepare(`
            SELECT COUNT(*) as count
            FROM domains 
            WHERE next_payment < date('now', '+30 days') AND status = 'active'
        `).get() as unknown;

        const stats: ComprehensiveStats = {
            // Customer stats
            totalCustomers: customerStats.total || 0,
            activeCustomers: customerStats.active || 0,
            leadCustomers: customerStats.leads || 0,
            inactiveCustomers: customerStats.inactive || 0,

            // Domain stats
            totalDomains: domainStats.total || 0,
            activeDomains: domainStats.active || 0,
            expiringDomains: expiringDomains.count || 0,
            expiredDomains: domainStats.expired || 0,

            // Project stats
            totalProjects: projectStats.total || 0,
            activeProjects: projectStats.active || 0,
            completedProjects: projectStats.completed || 0,
            overdueProjects: overdueProjects.count || 0,

            // Quote stats
            totalQuotes: quoteStats.total || 0,
            pendingQuotes: quoteStats.pending || 0,
            approvedQuotes: quoteStats.approved || 0,
            rejectedQuotes: quoteStats.rejected || 0,

            // Payment stats
            totalPayments: paymentStats.total || 0,
            pendingPayments: paymentStats.pending || 0,
            completedPayments: paymentStats.completed || 0,
            overduePayments: overduePayments.count || 0,
            totalRevenue: paymentStats.revenue || 0,

            // Activity stats
            totalActivities: activityStats.total || 0,
            pendingActivities: activityStats.pending || 0,
            completedActivities: activityStats.completed || 0,
            overdueActivities: activityStats.overdue || 0,

            // Reminder stats
            totalReminders: reminderStats.total || 0,
            pendingReminders: reminderStats.pending || 0,
            overdueReminders: reminderStats.overdue || 0,
            completedReminders: reminderStats.completed || 0,

            // Service stats
            totalEmailServices: emailServiceStats.total || 0,
            activeEmailServices: emailServiceStats.active || 0,
            totalTechnicalServices: technicalServiceStats.total || 0,
            activeTechnicalServices: technicalServiceStats.active || 0,
        };

        return NextResponse.json(stats);
    } catch (error) {
        console.error('Error fetching comprehensive stats:', error);
        return NextResponse.json({ error: 'Failed to fetch stats' }, { status: 500 });
    }
}
