import { NextRequest, NextResponse } from 'next/server';
import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { ProjectFormSchema } from '@/src/utils/comprehensiveValidators';

export async function GET(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const projects = db.prepare(`
            SELECT p.*, c.name as customer_name, c.email as customer_email
            FROM projects p
            LEFT JOIN customers c ON p.customer_id = c.id
            ORDER BY p.created_date DESC
        `).all();

        return NextResponse.json(projects);
    } catch (error) {
        console.error('Error fetching projects:', error);
        return NextResponse.json({ error: 'Failed to fetch projects' }, { status: 500 });
    }
}

export async function POST(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const body = await request.json();

        // Validate the request body
        const validatedData = ProjectFormSchema.parse(body);

        const id = `project_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
        const now = new Date().toISOString();

        const result = db.prepare(`
            INSERT INTO projects (
                id, customer_id, title, description, status, start_date, end_date,
                budget, progress, assigned_team, notes, created_date, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `).run(
            id,
            validatedData.customer_id,
            validatedData.title,
            validatedData.description || null,
            validatedData.status,
            validatedData.start_date || null,
            validatedData.end_date || null,
            validatedData.budget || null,
            validatedData.progress,
            validatedData.assigned_team || null,
            validatedData.notes || null,
            now,
            now
        );

        const newProject = db.prepare(`
            SELECT p.*, c.name as customer_name, c.email as customer_email
            FROM projects p
            LEFT JOIN customers c ON p.customer_id = c.id
            WHERE p.id = ?
        `).get(id);

        return NextResponse.json(newProject, { status: 201 });
    } catch (error) {
        console.error('Error creating project:', error);
        if (error instanceof Error && error.name === 'ZodError') {
            return NextResponse.json({ error: 'Validation error', details: error.message }, { status: 400 });
        }
        return NextResponse.json({ error: 'Failed to create project' }, { status: 500 });
    }
}
