import { NextRequest, NextResponse } from 'next/server';
import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { PaymentFormSchema } from '@/src/utils/comprehensiveValidators';

export async function GET(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const payments = db.prepare(`
            SELECT p.*, 
                   c.name as customer_name, 
                   d.domain_name,
                   pr.title as project_title,
                   q.title as quote_title
            FROM payments p
            LEFT JOIN customers c ON p.customer_id = c.id
            LEFT JOIN domains d ON p.domain_id = d.id
            LEFT JOIN projects pr ON p.project_id = pr.id
            LEFT JOIN quotes q ON p.quote_id = q.id
            ORDER BY p.created_date DESC
        `).all();

        return NextResponse.json(payments);
    } catch (error) {
        console.error('Error fetching payments:', error);
        return NextResponse.json({ error: 'Failed to fetch payments' }, { status: 500 });
    }
}

export async function POST(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const body = await request.json();

        // Validate the request body
        const validatedData = PaymentFormSchema.parse(body);

        const id = `payment_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
        const now = new Date().toISOString();

        const result = db.prepare(`
            INSERT INTO payments (
                id, customer_id, domain_id, project_id, quote_id, amount, payment_type,
                status, payment_method, payment_date, due_date, invoice_number,
                description, notes, created_date, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `).run(
            id,
            validatedData.customer_id,
            validatedData.domain_id || null,
            validatedData.project_id || null,
            validatedData.quote_id || null,
            validatedData.amount,
            validatedData.payment_type,
            validatedData.status,
            validatedData.payment_method,
            validatedData.payment_date || null,
            validatedData.due_date || null,
            validatedData.invoice_number || null,
            validatedData.description || null,
            validatedData.notes || null,
            now,
            now
        );

        const newPayment = db.prepare(`
            SELECT p.*, 
                   c.name as customer_name, 
                   d.domain_name,
                   pr.title as project_title,
                   q.title as quote_title
            FROM payments p
            LEFT JOIN customers c ON p.customer_id = c.id
            LEFT JOIN domains d ON p.domain_id = d.id
            LEFT JOIN projects pr ON p.project_id = pr.id
            LEFT JOIN quotes q ON p.quote_id = q.id
            WHERE p.id = ?
        `).get(id);

        return NextResponse.json(newPayment, { status: 201 });
    } catch (error) {
        console.error('Error creating payment:', error);
        if (error instanceof Error && error.name === 'ZodError') {
            return NextResponse.json({ error: 'Validation error', details: error.message }, { status: 400 });
        }
        return NextResponse.json({ error: 'Failed to create payment' }, { status: 500 });
    }
}
