import { NextRequest, NextResponse } from 'next/server';
import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { DomainFormSchema } from '@/src/utils/comprehensiveValidators';

export async function GET(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const domains = db.prepare(`
            SELECT d.*, c.name as customer_name, c.email as customer_email
            FROM domains d
            LEFT JOIN customers c ON d.customer_id = c.id
            ORDER BY d.created_date DESC
        `).all();

        return NextResponse.json(domains);
    } catch (error) {
        console.error('Error fetching domains:', error);
        return NextResponse.json({ error: 'Failed to fetch domains' }, { status: 500 });
    }
}

export async function POST(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const body = await request.json();

        // Validate the request body
        const validatedData = DomainFormSchema.parse(body);

        const id = `domain_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
        const now = new Date().toISOString();

        const result = db.prepare(`
            INSERT INTO domains (
                id, customer_id, domain_name, website, owner, email, package, 
                amount, status, priority, next_payment, notes, created_date, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `).run(
            id,
            validatedData.customer_id,
            validatedData.domain_name,
            validatedData.website || null,
            validatedData.owner || null,
            validatedData.email || null,
            validatedData.package || null,
            validatedData.amount || null,
            validatedData.status,
            validatedData.priority,
            validatedData.next_payment || null,
            validatedData.notes || null,
            now,
            now
        );

        const newDomain = db.prepare(`
            SELECT d.*, c.name as customer_name, c.email as customer_email
            FROM domains d
            LEFT JOIN customers c ON d.customer_id = c.id
            WHERE d.id = ?
        `).get(id);

        return NextResponse.json(newDomain, { status: 201 });
    } catch (error) {
        console.error('Error creating domain:', error);
        if (error instanceof Error && error.name === 'ZodError') {
            return NextResponse.json({ error: 'Validation error', details: error.message }, { status: 400 });
        }
        return NextResponse.json({ error: 'Failed to create domain' }, { status: 500 });
    }
}
