import { NextRequest, NextResponse } from 'next/server';
import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { CustomerFormSchema } from '@/src/utils/comprehensiveValidators';

export async function GET(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const customers = db.prepare('SELECT * FROM customers ORDER BY created_at DESC').all();

        return NextResponse.json(customers);
    } catch (error) {
        console.error('Error fetching customers:', error);
        return NextResponse.json({ error: 'Failed to fetch customers' }, { status: 500 });
    }
}

export async function POST(request: NextRequest) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const body = await request.json();

        // Validate the request body
        const validatedData = CustomerFormSchema.parse(body);

        const id = `customer_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
        const now = new Date().toISOString();

        const result = db.prepare(`
            INSERT INTO customers (
                id, name, email, phone, company, status, stage, 
                assigned_sales_rep, value, notes, created_at, 
                updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `).run(
            id,
            validatedData.name,
            validatedData.email || null,
            validatedData.phone || null,
            validatedData.company || null,
            validatedData.status,
            validatedData.stage,
            validatedData.assigned_sales_rep || null,
            validatedData.value,
            validatedData.notes || null,
            now,
            now
        );

        const newCustomer = db.prepare('SELECT * FROM customers WHERE id = ?').get(id);

        return NextResponse.json(newCustomer, { status: 201 });
    } catch (error) {
        console.error('Error creating customer:', error);
        if (error instanceof Error && error.name === 'ZodError') {
            return NextResponse.json({ error: 'Validation error', details: error.message }, { status: 400 });
        }
        return NextResponse.json({ error: 'Failed to create customer' }, { status: 500 });
    }
}
