import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

// GET /api/calendar - קבלת אירועים ומשימות ללוח שנה
export async function GET(request: NextRequest) {
    const db = getDb();
    try {
        const { searchParams } = new URL(request.url);
        const date = searchParams.get('date'); // YYYY-MM-DD
        const start_date = searchParams.get('start_date');
        const end_date = searchParams.get('end_date');
        const customer_id = searchParams.get('customer_id');
        const view = searchParams.get('view') || 'day'; // day, week, month

        let dateFilter = '';
        const params: any[] = [];

        if (date) {
            // תצוגת יום
            dateFilter = 'AND date(e.start_datetime) = ?';
            params.push(date);
        } else if (start_date && end_date) {
            // תצוגת טווח
            dateFilter = 'AND date(e.start_datetime) BETWEEN ? AND ?';
            params.push(start_date, end_date);
        } else {
            // ברירת מחדל: היום
            dateFilter = 'AND date(e.start_datetime) = date("now")';
        }

        if (customer_id) {
            dateFilter += ' AND e.customer_id = ?';
            params.push(customer_id);
        }

        // קבלת אירועים
        const events = db.prepare(`
            SELECT 
                e.id,
                e.title,
                e.description,
                e.start_datetime,
                e.end_datetime,
                e.location,
                e.event_type,
                e.status,
                e.customer_id,
                e.domain_id,
                e.project_id,
                e.tags,
                c.name as customer_name,
                c.company as customer_company,
                d.name as domain_name,
                p.name as project_title,
                'event' as type
            FROM events e
            LEFT JOIN customers c ON e.customer_id = c.id
            LEFT JOIN domains d ON e.domain_id = d.id
            LEFT JOIN projects p ON e.project_id = p.id
            WHERE 1=1 ${dateFilter}
            ORDER BY e.start_datetime ASC
        `).all(...params) as unknown as any[];

        // עיבוד תגיות מ-JSON
        events.forEach(event => {
            try {
                event.tags = event.tags ? JSON.parse(event.tags) : [];
            } catch (e) {
                event.tags = [];
            }
        });

        // קבלת משימות עם מועדים
        const tasks = db.prepare(`
            SELECT 
                t.id,
                t.title,
                t.description,
                t.due_date as start_datetime,
                t.due_date as end_datetime,
                NULL as location,
                'task' as event_type,
                t.status,
                t.priority,
                t.customer_id,
                t.domain_id,
                t.project_id,
                t.event_id,
                c.name as customer_name,
                c.company as customer_company,
                d.name as domain_name,
                p.name as project_title,
                e.title as event_title,
                'task' as type
            FROM tasks t
            LEFT JOIN customers c ON t.customer_id = c.id
            LEFT JOIN domains d ON t.domain_id = d.id
            LEFT JOIN projects p ON t.project_id = p.id
            LEFT JOIN events e ON t.event_id = e.id
            WHERE t.due_date IS NOT NULL ${dateFilter.replace('e.start_datetime', 't.due_date')}
            ORDER BY t.due_date ASC
        `).all(...params) as unknown as any[];

        // סטטיסטיקות
        const stats = {
            events_count: events.length,
            tasks_count: tasks.length,
            total_count: events.length + tasks.length,
            events_by_type: {} as Record<string, number>,
            tasks_by_status: {} as Record<string, number>,
            tasks_by_priority: {} as Record<string, number>
        };

        // ספירת אירועים לפי סוג
        events.forEach(event => {
            const type = event.event_type || 'other';
            stats.events_by_type[type] = (stats.events_by_type[type] || 0) + 1;
        });

        // ספירת משימות לפי סטטוס ועדיפות
        tasks.forEach(task => {
            const status = task.status || 'pending';
            const priority = task.priority || 'medium';
            stats.tasks_by_status[status] = (stats.tasks_by_status[status] || 0) + 1;
            stats.tasks_by_priority[priority] = (stats.tasks_by_priority[priority] || 0) + 1;
        });

        return NextResponse.json({
            events,
            tasks,
            stats,
            date: date || new Date().toISOString().split('T')[0],
            view
        });

    } catch (e: unknown) {
        console.error('Get calendar data error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת נתוני לוח שנה: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}
